/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.request;

import java.time.Instant;
import java.util.Locale;
import java.util.Map;
import java.util.stream.Stream;

import javax.annotation.Nonnull;

import com.sap.cds.services.impl.utils.ModifiedValues;
import com.sap.cds.services.impl.utils.ModifiedValues.MapProperty;
import com.sap.cds.services.impl.utils.ModifiedValues.Property;
import com.sap.cds.services.request.ModifiableParameterInfo;
import com.sap.cds.services.request.ParameterInfo;
import com.sap.cds.services.utils.TemporalRangeUtils;

class ModifiableParameterInfoImpl implements ModifiableParameterInfo {

	private final ModifiedValues modifyValues = new ModifiedValues();

	private final MapProperty<String, String> headers;

	private final MapProperty<String, String> queryParams;

	private final Property<Locale> locale;

	private final Property<Instant> validFrom;

	private final Property<Instant> validTo;

	ModifiableParameterInfoImpl(@Nonnull ParameterInfo prevParameterInfo) {

		this.locale = modifyValues.field("locale", prevParameterInfo::getLocale);
		this.headers = modifyValues.field("headers", prevParameterInfo::getHeaders, prevParameterInfo::getHeader);
		this.validFrom = modifyValues.field("validFrom", prevParameterInfo::getValidFrom);
		this.validTo = modifyValues.field("validTo", prevParameterInfo::getValidTo);
		this.queryParams = modifyValues.field("queryParams", prevParameterInfo::getQueryParams, prevParameterInfo::getQueryParameter);
	}

	@Override
	public Map<String, String> getHeaders() {
		return headers.getValue();
	}

	@Override
	public ModifiableParameterInfo setHeaders(Map<String, String> val) {
		headers.modifyValue(val);
		return this;
	}

	@Override
	public String getHeader(String id) {
		return headers.getMapValue(id);
	}

	@Override
	public ModifiableParameterInfo setHeader(String id, String value) {
		headers.modifyMapValue(id, value);
		return this;
	}

	@Override
	public Map<String, String> getQueryParams() {
		return queryParams.getValue();
	}

	@Override
	public ModifiableParameterInfo setQueryParameters(Map<String, String> val) {
		queryParams.modifyValue(val);
		return this;
	}

	@Override
	public String getQueryParameter(String key) {
		return queryParams.getMapValue(key);
	}

	@Override
	public ModifiableParameterInfo setQueryParameter(String key, String value) {
		queryParams.modifyMapValue(key, value);
		return this;
	}

	@Override
	public Stream<String> getQueryParameters() {
		return getQueryParams().keySet().stream();
	}

	@Override
	public Locale getLocale() {
		return locale.getValue();
	}

	@Override
	public ModifiableParameterInfo setLocale(Locale val) {
		locale.modifyValue(val);
		return this;
	}

	@Override
	public Instant getValidFrom() {
		return validFrom.getValue();
	}

	@Override
	public ModifiableParameterInfo setValidFrom(Instant val) {
		validFrom.modifyValue(val);
		return this;
	}

	@Override
	public Instant getValidTo() {
		return validTo.getValue();
	}

	@Override
	public ModifiableParameterInfo setValidTo(Instant val) {
		validTo.modifyValue(val);
		return this;
	}

	@Override
	public ModifiableParameterInfo setValidAt(Instant validAt) {
		Instant[] range = TemporalRangeUtils.getTemporalRanges(validAt);
		setValidFrom(range[0]);
		setValidTo(range[1]);
		return this;
	}

}
