/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.request;

import java.sql.Timestamp;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoField;
import java.time.temporal.TemporalAccessor;
import java.util.Locale;
import java.util.Map;
import java.util.TreeMap;
import java.util.stream.Stream;

import com.sap.cds.services.request.ParameterInfo;
import com.sap.cds.services.runtime.RequestParameters;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.HttpHeaders;
import com.sap.cds.services.utils.LocaleUtils;
import com.sap.cds.services.utils.QueryParameters;
import com.sap.cds.services.utils.StringUtils;

@Deprecated
public class ParameterInfoImpl implements ParameterInfo {
	// smallest instant that worked in tests on a HANA DB
	private static final Instant MIN_VALID_FROM = Instant.parse("0001-01-01T00:00:00Z");

	private static final Instant MAX_VALID_TO = Instant.parse("9999-12-31T23:59:59.999999999Z");

	private final RequestParameters requestParameter;

	private Locale locale;

	private Instant validFrom;

	private Instant validTo;

	public ParameterInfoImpl(RequestParameters requestParameter) {
		this.requestParameter = requestParameter;
	}

	@Override
	public String getHeader(String id) {
		return requestParameter.getHeader(id);
	}

	@Override
	public String getQueryParameter(String key) {
		return requestParameter.getQueryParameter(key);
	}

	@Override
	public Stream<String> getQueryParameters() {
		return requestParameter.getQueryParameters();
	}

	@Override
	public Map<String, String> getHeaders() {
		throw new UnsupportedOperationException("RequestParameters is deprecated and does not support getHeaders()");
	}

	@Override
	public Map<String, String> getQueryParams() {
		Map<String, String> result = new TreeMap<>();
		getQueryParameters().forEach(q -> result.put(q, getQueryParameter(q)));
		return result;
	}

	@Override
	public Locale getLocale() {
		if (locale == null) {
			locale = LocaleUtils.getLocale(
					getQueryParameter(QueryParameters.SAP_LANGUAGE),
					getHeader(HttpHeaders.X_SAP_REQUEST_LANGUAGE_HEADER),
					getHeader(HttpHeaders.ACCEPT_LANGUAGE));
		}
		return locale;
	}

	@Override
	public Instant getValidFrom() {
		if (this.validFrom == null) {
			setTemporalRange();
		}
		return this.validFrom;
	}

	@Override
	public Instant getValidTo() {
		if (this.validTo == null) {
			setTemporalRange();
		}
		return this.validTo;
	}

	// private helpers
	private void setTemporalRange() {
		this.validFrom = parseDate(getQueryParameter(QueryParameters.VALID_FROM), QueryParameters.VALID_FROM);
		this.validTo = parseDate(getQueryParameter(QueryParameters.VALID_TO), QueryParameters.VALID_TO);

		if (this.validFrom == null && this.validTo == null) {
			Instant validAt = parseDate(getQueryParameter(QueryParameters.VALID_AT), QueryParameters.VALID_AT);
			if (validAt == null) {
				// use now() if parameter valid-at isn't provided:
				// https://github.wdf.sap.corp/pages/cap/guides/temporal-data#reading-temporal-data
				this.validFrom = Instant.now();
			} else {
				this.validFrom = validAt;
			}
			this.validTo = this.validFrom.plusMillis(1);
			return;
		}

		if (this.validFrom == null) {
			this.validFrom = MIN_VALID_FROM;
		}

		if (this.validTo == null) {
			this.validTo = MAX_VALID_TO;
		}
	}

	private static Instant parseDate(String date, String hint) {
		if (!StringUtils.isEmpty(date)) {
			try {
				// expects date-time in format: '2011-12-03T10:15:30' or
				// '2011-12-03T10:15:30+01:00'
				return Instant.from(DateTimeFormatter.ISO_OFFSET_DATE_TIME.parse(date));
			} catch (DateTimeParseException e1) {
				try {
					// expects just date in format: '2011-12-03' or '2011-12-03+01:00'
					TemporalAccessor accessor = DateTimeFormatter.ISO_DATE.parse(date);

					// extract zone offset from parsed date
					ZoneOffset zoneOffset = getZoneOffset(accessor);

					// since there is only a date, we need to guess the time according the hint
					LocalTime time = getLocalTime(hint);

					// add time to a have a complete date-time and get it converted to instant
					LocalDateTime dateTime = LocalDate.from(accessor).atTime(time);

					// use given time offset for creating instant
					return dateTime.toInstant(zoneOffset);
				} catch (DateTimeParseException e2) { // NOSONAR
					// try Timestamp format as fallback
					try {
						return Timestamp.valueOf(date).toLocalDateTime().toInstant(ZoneOffset.UTC);
					} catch (IllegalArgumentException e3) { // NOSONAR
						throw new ErrorStatusException(CdsErrorStatuses.INVALID_DATE_VALUE, date, hint, e1);
					}
				}
			}
		}
		return null;
	}

	private static ZoneOffset getZoneOffset(TemporalAccessor accessor) {
		if (accessor.isSupported(ChronoField.OFFSET_SECONDS)) {
			return ZoneOffset.ofTotalSeconds(accessor.get(ChronoField.OFFSET_SECONDS));
		}
		// return UTC if given temporal object doesn't provide an offset field
		return ZoneOffset.UTC;
	}

	private static LocalTime getLocalTime(String hint) {
		switch (hint) {
		case QueryParameters.VALID_FROM:
		case QueryParameters.VALID_AT:
		default:
			return LocalTime.MIN;
		case QueryParameters.VALID_TO:
			return LocalTime.MAX;
		}
	}
}
