/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.runtime;

import java.util.function.Function;
import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.feature.auth.AuthenticatedUserClaimProvider;
import com.sap.cds.feature.auth.AuthenticatedUserInfoProvider;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.ErrorStatuses;
import com.sap.cds.services.Service;
import com.sap.cds.services.changeset.ChangeSetContext;
import com.sap.cds.services.impl.ServiceCatalogImpl;
import com.sap.cds.services.impl.ServiceCatalogSPI;
import com.sap.cds.services.impl.ServiceSPI;
import com.sap.cds.services.impl.handlerregistry.HandlerRegistryTools;
import com.sap.cds.services.impl.request.ParameterInfoImpl;
import com.sap.cds.services.impl.request.UserInfoImpl;
import com.sap.cds.services.impl.utils.CdsModelUtils;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.request.ParameterInfo;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.request.UserInfo;
import com.sap.cds.services.runtime.CdsModelProvider;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.runtime.ChangeSetContextRunner;
import com.sap.cds.services.runtime.ParameterInfoProvider;
import com.sap.cds.services.runtime.Request;
import com.sap.cds.services.runtime.RequestContextRunner;
import com.sap.cds.services.runtime.UserInfoProvider;
import com.sap.cds.services.utils.ErrorStatusException;

/**
 * Bootstrap for setting up service backbone. This includes reading the CDS
 * model, creating corresponding services and registering default and custom
 * handlers.
 */
@SuppressWarnings("deprecation")
public class CdsRuntimeImpl implements CdsRuntime {

	private final static Logger logger = LoggerFactory.getLogger(CdsRuntimeImpl.class);
	private final static String basePackage = Service.class.getPackage().getName();

	private final ServiceCatalogImpl serviceCatalog = new ServiceCatalogImpl();
	private CdsModel cdsModel = CdsModelUtils.loadCdsModel(null); // empty model

	private CdsModelProvider cdsModelProvider = this::tenantIndependentModelProvider;
	private UserInfoProvider userInfoProvider = CdsRuntimeImpl::authenticatedUserInfo;
	private ParameterInfoProvider parameterInfoProvider = ParameterInfo::create; // will be overridden by app framework

	CdsRuntimeImpl() {
		// only visible in this package
	}

	@Override
	public ServiceCatalogSPI getServiceCatalog() {
		return serviceCatalog;
	}

	@Override
	public CdsModel getCdsModel() {
		return cdsModel;
	}

	@Override
	public CdsModel getCdsModel(String tenant) {
		if(tenant == null) {
			return getCdsModel();
		}

		return cdsModelProvider.get(tenant);
	}

	@Override
	public <T> T runInRequestContext(Request request, Function<RequestContext, T> requestHandler) {
		RequestContextRunner runInRequestContext = requestContext();
		if (request != null && request.getRequestUser() != null) {
			runInRequestContext.user( new UserInfoImpl(request.getRequestUser()) );
		}
		if (request != null && request.getRequestParameters() != null) {
			runInRequestContext.parameters(new ParameterInfoImpl(request.getRequestParameters()));
		}
		return runInRequestContext.run(requestHandler);
	}

	@Override
	public RequestContextRunner requestContext() {
		return new RequestContextRunnerImpl(this);
	}

	@Override
	public <T> T runInChangeSetContext(Function<ChangeSetContext, T> changeSetHandler) {
		return changeSetContext().run(changeSetHandler);
	}

	@Override
	public ChangeSetContextRunner changeSetContext() {
		return new ChangeSetContextRunnerImpl();
	}

	// These methods are made available through CdsRuntimeConfigurer
	// They are therefore NOT public, but only visible in this package

	void setCdsModel(String csnPath) {
		this.cdsModel = CdsModelUtils.loadCdsModel(csnPath);
	}

	void setCdsModel(CdsModel model) {
		this.cdsModel = model;
	}

	void setCdsModelProvider(CdsModelProvider provider) {
		cdsModelProvider = provider;
	}

	CdsModelProvider getCdsModelProvider() {
		return cdsModelProvider;
	}

	void setUserInfoProvider(UserInfoProvider userInfoProvider) {
		this.userInfoProvider = userInfoProvider;
	}

	UserInfoProvider getUserInfoProvider() {
		return userInfoProvider;
	}

	void setParameterInfoProvider(ParameterInfoProvider parameterInfoProvider) {
		this.parameterInfoProvider = parameterInfoProvider;
	}

	ParameterInfoProvider getParameterInfoProvider() {
		return parameterInfoProvider;
	}

	@Override
	public ParameterInfo getProvidedParameterInfo() {
		ParameterInfo parameterInfo = parameterInfoProvider.get();
		return parameterInfo != null ? parameterInfo : ParameterInfo.create();
	}

	@Override
	public UserInfo getProvidedUserInfo() {
		UserInfo userInfo = userInfoProvider.get();
		return userInfo != null ? userInfo : UserInfo.create();
	}

	void registerService(Service service) {
		serviceCatalog.register(service);
		ServiceSPI serviceSPI = CdsServiceUtils.getServiceSPI(service);
		if(serviceSPI != null) {
			serviceSPI.setCdsRuntime(this);
		}
	}

	<T> void registerEventHandler(Class<T> handlerClass, Supplier<T> handlerFactory) {
		HandlerRegistryTools.registerClass(handlerClass, handlerFactory, serviceCatalog);
		if(handlerClass.getPackage().getName().startsWith(basePackage)) {
			logger.debug("Registered handler class {}", handlerClass.getName());
		} else {
			logger.info("Registered handler class {}", handlerClass.getName());
		}
	}

	private CdsModel tenantIndependentModelProvider(String tenant) {
		return cdsModel;
	}

	private static UserInfo authenticatedUserInfo() {
		String claim = AuthenticatedUserClaimProvider.INSTANCE.getUserClaim();
		if (claim != null) {
			try {
				return AuthenticatedUserInfoProvider.INSTANCE.extract(claim);
			} catch(Exception e) {  // NOSONAR
				throw new ErrorStatusException(ErrorStatuses.UNAUTHORIZED, e); // if there is a claim the user must be extracted
			}
		}
		return null; // e.g. anonymous access
	}

}
