/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.runtime;

import static com.sap.cds.services.impl.request.ParameterInfoFactoryImpl.emptyParameterInfo;
import static com.sap.cds.services.impl.request.UserInfoFactoryImpl.anonymousUserInfo;
import static com.sap.cds.services.impl.request.UserInfoFactoryImpl.privilegedUserInfo;

import java.util.Objects;
import java.util.function.Consumer;
import java.util.function.Function;

import javax.annotation.Nullable;

import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.ServiceCatalog;
import com.sap.cds.services.impl.ServiceCatalogImpl;
import com.sap.cds.services.impl.messages.MessagesImpl;
import com.sap.cds.services.impl.request.RequestContextImpl;
import com.sap.cds.services.impl.request.RequestContextSPI;
import com.sap.cds.services.messages.Messages;
import com.sap.cds.services.request.ModifiableParameterInfo;
import com.sap.cds.services.request.ModifiableUserInfo;
import com.sap.cds.services.request.ParameterInfo;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.request.UserInfo;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.runtime.RequestContextRunner;


public class RequestContextRunnerImpl implements RequestContextRunner {

	/**
	 * A reference to the active runtime. Should be <code>null</code> in test scope only.
	 */
	private final CdsRuntime cdsRuntime;
	private final RequestContextSPI currentRequestContext;

	private UserInfo userInfo;
	private ParameterInfo parameterInfo;
	private boolean clearMessages = false;

	public RequestContextRunnerImpl(@Nullable CdsRuntime cdsRuntime) {
		this.cdsRuntime = cdsRuntime;
		this.currentRequestContext = RequestContextImpl.getCurrentInternal();

		// ensures that provider-dependant info objects are always initialized as part of the parent thread
		// this comes at the cost of being potentially unnecessary, in case UserInfo or ParameterInfo objects are directly passed to the runner
		// however the benefit of usability of the API outweighs these drawbacks
		if (currentRequestContext != null) {
			userInfo = currentRequestContext.getUserInfo();
			parameterInfo = currentRequestContext.getParameterInfo();
		} else {
			userInfo = providedUserInfo(); // might return null
			if (userInfo == null) {
				userInfo = anonymousUserInfo();
			}

			parameterInfo = providedParameterInfo(); // might return null
			if(parameterInfo == null) {
				parameterInfo = emptyParameterInfo();
			}
		}
		assert userInfo != null;
		assert parameterInfo != null;
	}

	@Override
	public RequestContextRunner user(UserInfo userInfo) {
		Objects.requireNonNull(userInfo, "userInfo must not be null");
		this.userInfo = userInfo;
		return this;
	}

	@Override
	public RequestContextRunner anonymousUser() {
		userInfo = anonymousUserInfo();
		return this;
	}

	@Override
	public RequestContextRunner providedUser() {
		userInfo = providedUserInfo();
		return this;
	}

	@Override
	public RequestContextRunner privilegedUser() {
		userInfo = privilegedUserInfo();
		return this;
	}

	@Override
	public RequestContextRunner modifyUser(Consumer<ModifiableUserInfo> contextUser) {
		UserInfo prevUserInfo = userInfo;

		ModifiableUserInfo modifiedUser = prevUserInfo.copy();
		if (contextUser != null) {
			contextUser.accept(modifiedUser);
		}
		userInfo = modifiedUser;

		return this;
	}

	@Override
	public RequestContextRunner parameters(ParameterInfo parameterInfo) {
		Objects.requireNonNull(parameterInfo, "parameterInfo must not be null");
		this.parameterInfo = parameterInfo;
		return this;
	}

	@Override
	public RequestContextRunner clearParameters() {
		parameterInfo = emptyParameterInfo();
		return this;
	}

	@Override
	public RequestContextRunner providedParameters() {
		parameterInfo = providedParameterInfo();
		return this;
	}

	@Override
	public RequestContextRunner modifyParameters(Consumer<ModifiableParameterInfo> contextParamters) {
		ParameterInfo prevParameterInfo = parameterInfo;

		ModifiableParameterInfo modifiedParameter = prevParameterInfo.copy();
		if (contextParamters != null) {
			contextParamters.accept(modifiedParameter);
		}
		parameterInfo = modifiedParameter;

		return this;
	}

	@Override
	public RequestContextRunner clearMessages() {
		clearMessages = true;
		return this;
	}

	@Override
	public void run(Consumer<RequestContext> requestHandler) {
		run( (requestContext) -> {
			requestHandler.accept(requestContext);
			return Void.TYPE;
		});
	}

	@Override
	public <T> T run(Function<RequestContext, T> requestHandler) {
		CdsModel cdsModel;
		ServiceCatalog serviceCatalog;
		if (cdsRuntime != null) {
			serviceCatalog = cdsRuntime.getServiceCatalog(); // could be exposed by a provider later
			cdsModel = cdsRuntime.getCdsModel(userInfo.getTenant());
		} else {
			// only test scope
			serviceCatalog = new ServiceCatalogImpl();
			cdsModel = null; // CdsModelBuilder.create().build();
		}

		Messages messages;
		if (!clearMessages && currentRequestContext != null) {
			messages = currentRequestContext.getMessages();
		} else {
			messages = new MessagesImpl(parameterInfo.getLocale());
		}

		try (RequestContextSPI requestContext = RequestContextImpl.open(cdsModel, serviceCatalog, userInfo, parameterInfo, messages)) {
			return requestHandler.apply(requestContext);
		}
	}

	private UserInfo providedUserInfo() {
		if (cdsRuntime != null) {
			return cdsRuntime.getProvidedUserInfo();
		}
		return null;
	}

	private ParameterInfo providedParameterInfo() {
		if (cdsRuntime != null) {
			return cdsRuntime.getProvidedParameterInfo();
		}
		return null;
	}

}
