/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl;

import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.services.Service;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;

public class ServiceCatalogImpl implements ServiceCatalogSPI {

	private final static Logger logger = LoggerFactory.getLogger(ServiceCatalogImpl.class);

	private final Map<String, Service> registeredServices = new ConcurrentHashMap<>();

	@Override
	public void register(Service service) {
		Objects.requireNonNull(service, "service is required");

		String serviceName = service.getName();
		if (registeredServices.putIfAbsent(serviceName, service) != null) {
			throw new ErrorStatusException(CdsErrorStatuses.DUPLICATE_SERVICE, serviceName);
		}

		logger.info("Registered service {}", serviceName);
		logger.debug("{} has type '{}'", serviceName, service.getClass().getName());
	}

	@Override
	public Service getService(String name) {
		return registeredServices.get(name);
	}

	@Override
	@SuppressWarnings("unchecked")
	public <S extends Service> Stream<S> getServices(Class<S> serviceClass) {
		Objects.requireNonNull(serviceClass, "Class of the service is required");

		return registeredServices.values().stream().filter(s -> serviceClass.isInstance(s)).map(s -> (S) s);
	}

	@Override
	@SuppressWarnings("unchecked")
	public <S extends Service> S getService(Class<S> serviceClass, String name) {
		Objects.requireNonNull(serviceClass, "Class of the service is required");

		Service service = getService(name);
		if (service != null && serviceClass.isInstance(service)) {
			return (S) service;
		}
		return null;
	}

	@Override
	public Stream<Service> getServices() {
		return registeredServices.values().stream();
	}
}
