/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.authorization;

import java.util.List;
import java.util.stream.Collectors;

import com.sap.cds.ql.CQL;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.authorization.ActionAccessEventContext;
import com.sap.cds.services.authorization.AuthorizationService;
import com.sap.cds.services.authorization.CalcWhereConditionEventContext;
import com.sap.cds.services.authorization.EntityAccessEventContext;
import com.sap.cds.services.authorization.FunctionAccessEventContext;
import com.sap.cds.services.authorization.ServiceAccessEventContext;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.StringUtils;
import com.sap.cds.services.utils.TenantAwareCache;
import com.sap.cds.services.utils.model.Privilege;
import com.sap.cds.services.utils.model.Restriction;

@ServiceName(value = "*", type = AuthorizationService.class)
public class AuthorizationDefaultOnHandler implements EventHandler {

	private final TenantAwareCache<RestrictionLookup, CdsModel> restrictionLookupCache;
	private final TenantAwareCache<PredicateLookup, CdsModel> predicateLookupCache;

	AuthorizationDefaultOnHandler(CdsRuntime runtime) {
		restrictionLookupCache = TenantAwareCache.create(() -> new RestrictionLookup(), runtime);
		predicateLookupCache = TenantAwareCache.create(() -> new PredicateLookup(), runtime);
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected void defaultHasServiceAccess(ServiceAccessEventContext context) {

		final String event = context.getAccessEventName();
		final String service = context.getAccessServiceName();

		boolean result = true; // as long as there is no restriction access is granted

		Restriction restriction = restrictionLookupCache.findOrCreate().retrieveServiceRestriction(context.getModel(), service);
		if (restriction != null) {
			result = RestrictionUtils.passesRestriction(restriction, context.getUserInfo(), event);
		}

		context.setResult(result);
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected void defaultHasEntityAccess(EntityAccessEventContext context) {

		final String event = context.getAccessEventName();
		final String entityName = context.getAccessEntityName();

		boolean result = true; // as long as there is no restriction access is granted

		Restriction restriction = restrictionLookupCache.findOrCreate().retrieveEntityRestriction(context.getModel(), entityName);
		if (restriction != null) {
			result = RestrictionUtils.passesRestriction(restriction, context.getUserInfo(), event);
		}

		context.setResult(result);
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected void defaultHasFunctionAccess(FunctionAccessEventContext context) {

		final String functionName = context.getFunctionName();
		final String entityName = context.getEntityName();

		boolean result = true; // as long as there is no restriction access is granted

		Restriction restriction = restrictionLookupCache.findOrCreate().lookupFunctionRestriction(context.getModel(), entityName, functionName);
		if (restriction != null) {
			result = RestrictionUtils.passesRestriction(restriction, context.getUserInfo(), Privilege.PredefinedGrant.ALL.toString() /* = event: restrictions on functions have implicitly grant '*' */);
		}

		context.setResult(result);
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected void defaultHasActionAccess(ActionAccessEventContext context) {

		final String actionName = context.getActionName();
		final String entityName = context.getEntityName();

		boolean result = true; // as long as there is no restriction access is granted

		Restriction restriction = restrictionLookupCache.findOrCreate().lookupActionRestriction(context.getModel(), entityName, actionName);
		if (restriction != null) {
			result = RestrictionUtils.passesRestriction(restriction, context.getUserInfo(), Privilege.PredefinedGrant.ALL.toString() /* = event: restrictions on actions have implicitly grant '*' */);
		}

		context.setResult(result);
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected void defaultCalcWhereCondition(CalcWhereConditionEventContext context) {

		final String event = context.getEventName();
		final String entityName = context.getEntityName();

		CqnPredicate result = null; // no filter as long there is no where condition hit

		if ( !context.getUserInfo().isPrivileged() ) {
			// we can't add where clauses to CREATE / UPSERT events!
			Restriction restriction = restrictionLookupCache.findOrCreate().retrieveEntityRestriction(context.getModel(), entityName);
			if (restriction != null) {
				List<Privilege> privileges = RestrictionUtils.passingPrivilegesOfRestriction(restriction, context.getUserInfo(), event).collect(Collectors.toList());
				for (Privilege privilege : privileges) {
					CqnPredicate cqnPredicate = null;
					if ( !StringUtils.isEmpty(privilege.getCxnWhereCondition()) ) {
						try {
							cqnPredicate = predicateLookupCache.findOrCreate().resolvePredicate(privilege.getCxnWhereCondition(), context.getUserInfo());
						} catch (Exception e) { // NOSONAR
							throw new ErrorStatusException(CdsErrorStatuses.INVALID_WHERE_CONDITION, privilege.getCxnWhereCondition(), entityName, event, context.getUserInfo().getName(), e);
						}
					} else if( !StringUtils.isEmpty(privilege.getWhereCondition()) ) {
						// no _where property (Cxn) generated for where clause, e.g. compilation error silently ignored
						throw new ErrorStatusException(CdsErrorStatuses.INCONSISTENT_WHERE_CONDITION, privilege.getWhereCondition(), entityName);
					}

					if (cqnPredicate == null) { // null evaluates to 'true'
						result = null;
						break;
					}
					result = (result != null) ? CQL.or(result, cqnPredicate) : cqnPredicate;
				}
			}
		}

		context.setResult(result);
	}

}
