/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.cds;

import static com.sap.cds.services.utils.model.CdsAnnotations.ASSERT_RANGE;

import java.util.stream.Collectors;

import com.sap.cds.CdsDataProcessor;
import com.sap.cds.ql.cqn.Path;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEnumType;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.cds.CdsService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;

@ServiceName(value = "*", type = ApplicationService.class)
public class EnumAssertionHandler implements EventHandler {

	@Before(event = { CdsService.EVENT_CREATE, CdsService.EVENT_UPDATE, CdsService.EVENT_UPSERT })
	@HandlerOrder(OrderConstants.Before.VALIDATE_FIELDS)
	public void runCheck(EventContext context) {
		CdsDataProcessor.create()
				.addValidator(EnumAssertionHandler::isEnumWithAssertRange, EnumAssertionHandler::ensureValidEnum)
				.process(CdsServiceUtils.getEntities(context), context.getTarget());
	}

	private static boolean isEnumWithAssertRange(Path path, CdsElement element, CdsType type) {
		return type.isEnum() && !Boolean.FALSE.equals(ASSERT_RANGE.getOrDefault(element));
	}

	private static void ensureValidEnum(Path path, CdsElement element, Object elementValue) {
		if (elementValue != null) {
			CdsEnumType<?> enumElement = element.getType();
			if (!enumElement.hasValue(elementValue)) {
				String validEnumValues = enumElement.enumerals().values().stream().map(e -> e.value() + " (" + e.name() + ")").collect(Collectors.joining(", "));
				throw new ErrorStatusException(CdsErrorStatuses.INVALID_ENUM, elementValue, element.getName(), validEnumValues);
			}
		}
	}

}
