/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.messaging.message;

import java.io.IOException;
import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Map;
import java.util.UUID;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.sap.cds.feature.config.Properties;
import com.sap.cds.feature.config.pojo.CdsProperties.Application.ApplicationServiceConfig;
import com.sap.cds.feature.platform.ApplicationProperties;
import com.sap.cds.feature.platform.PlatformEnvironment;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;

/**
 * The implementation of the CDS structured JSON formatted message. The message structure is define in
 * the SAP TG27 (https://github.tools.sap/CentralEngineering/TechnologyGuidelines/blob/latest/tg27/docs/spec.md#subject-attribute)
 */
public class CdsMessage {

	private static ObjectMapper mapper = new ObjectMapper().configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);

	/**
	 * Factory method for creating an instance of the CDS message.
	 *
	 * @param service full qualified service name
	 * @param event the event name
	 * @param data the structured payload data of the message
	 * @return the {@link CdsMessage}
	 */
	public static CdsMessage create(String service, String event, Map<String, Object> data) {


		CdsMessage result = new CdsMessage();

		result.specversion = "1.0";
		result.datacontenttype = "application/json";

		result.time = DateTimeFormatter.ISO_OFFSET_DATE_TIME.format(Instant.now().atZone(ZoneId.of("UTC")));
		result.id = UUID.randomUUID().toString();

		ApplicationServiceConfig serviceCfg = Properties.getCds().getApplication().getService(service);

		if (serviceCfg != null) {
			result.source = serviceCfg.getSource();
		}

		if (result.source == null) {
			ApplicationProperties appProperties = PlatformEnvironment.INSTANCE.getApplicationProperties();
			result.source = "/default/" + appProperties.getApplicationName();
		}

		result.type = service + '.' + event;

		// TODO: sap passport integration

		result.data = data;

		return result;
	}

	/**
	 * Parses the {@link CdsMessage} from the given raw message
	 *
	 * @param jsonString the raw JSON message
	 * @return the {@link CdsMessage}
	 */
	public static CdsMessage parse(String jsonString) {
		try {
			return mapper.readValue(jsonString, CdsMessage.class);
		} catch (IOException e) {
			throw new ErrorStatusException(CdsErrorStatuses.INVALID_CDS_MESSAGE, e);
		}
	}

	@JsonProperty("specversion")
	private String specversion;

	@JsonProperty("type")
	private String type;

	@JsonProperty("source")
	private String source;

	@JsonProperty("subject")
	private String subject;

	@JsonProperty("id")
	private String id;

	@JsonProperty("datacontenttype")
	private String datacontenttype;

	@JsonProperty("dataschema")
	private String dataschema;

	@JsonProperty("sappassport")
	private String sappassport;

	@JsonProperty("time")
	private String time;

	@JsonProperty("data")
	private Map<String, Object> data;


	/**
	 * Creates the CDS event context from the message.
	 *
	 * @return event context
	 */
	public CdsMessageEventContext toCdsMessageEventContext() {

		String event = type;

		if (type.indexOf('.') !=  -1) {
			event = type.substring(type.lastIndexOf('.') + 1);
		}

		CdsMessageEventContext ctx = CdsMessageEventContext.create(event);

		ctx.setSpecVersion(specversion);
		ctx.setType(type);
		ctx.setSource(source);
		ctx.setSubject(subject);
		ctx.setId(id);
		ctx.setDataContentType(datacontenttype);
		ctx.setDataSchema(dataschema);
		ctx.setSapPassport(sappassport);
		ctx.setTime(Instant.from(DateTimeFormatter.ISO_OFFSET_DATE_TIME.parse(time)));
		ctx.setData(data);
		return ctx;
	}
}
