/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.persistence;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UncheckedIOException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Charsets;
import com.sap.cds.feature.config.Properties;
import com.sap.cds.ql.Insert;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsSimpleType;
import com.sap.cds.services.ServiceException;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.StringUtils;
import com.sap.cds.util.CdsTypeUtils;

public class CsvDataLoader {

	private final static Logger logger = LoggerFactory.getLogger(CsvDataLoader.class);
	private final static String FILE_SUFFIX = Properties.getCds().getDataSource().getCsvFileSuffix();

	private final PersistenceService db;
	private final CdsModel model;

	public CsvDataLoader(PersistenceService db, CdsModel model) {
		this.db = db;
		this.model = model;
	}

	/**
	 * Loads CSVs from the first existing CSV folder location.
	 * The paths is resolved based on the current working directory
	 */
	public void load() {
		for(String path : Properties.getCds().getDataSource().getCsvPaths()) {
			boolean recursive = false;
			if(path.endsWith("**") && path.length() > 2) {
				path = path.substring(0, path.length() - 2);
				recursive = true;
			}

			File folder = new File(path);
			if(folder.exists() && folder.isDirectory()) {
				loadFolder(folder, recursive);
			}
		}
	}

	/**
	 * Loads CSV files from within the specified folder
	 * @param folder the folder
	 * @param recursive true, if the folder should be scanned recursively
	 */
	public void loadFolder(File folder, boolean recursive) {
		for (File file : folder.listFiles()) {
			if (file.isFile() && file.getName().endsWith(FILE_SUFFIX)) {
				loadFile(file);
			}

			if (recursive && file.isDirectory()) {
				loadFolder(file, true);
			}
		}
	}

	/**
	 * Loads CSV files from the specified file (extension needs to be .csv)
	 * @param file the file
	 */
	public void loadFile(File file) {
		try (CSV csv = new CSV(file)) {
			List<Map<String, Object>> entries = csv.data().collect(Collectors.toList());
			if(!entries.isEmpty()) {
				Insert insert = Insert.into(csv.entity()).entries(entries);
				try {
					db.run(insert);
					logger.info("Filling {} from {}", csv.entity().getQualifiedName(), file.getPath());
				} catch (ServiceException e) {
					logger.debug("Skipped filling {} from {}", csv.entity.getQualifiedName(), file.getPath(), e);
				}
			}
		} catch (UncheckedIOException | IOException e) {
			throw new ErrorStatusException(CdsErrorStatuses.INVALID_CSV_FILE, file.getPath(), e); // NOSONAR
		}
	}

	private class CSV implements AutoCloseable {

		private final CdsEntity entity;
		private final File file;
		private final BufferedReader br;
		private final List<CdsElement> headers;

		private String separator = ";";

		public CSV(File file) throws IOException {
			this.file = file;
			br = new BufferedReader(new InputStreamReader(new FileInputStream(file), Charsets.UTF_8));

			String entityName = entityName(file.getName());
			entity = model.findEntity(entityName).orElse(null);
			if(entity == null) {
				throw new ErrorStatusException(CdsErrorStatuses.INVALID_CSV_FILE_ENTITYNOTFOUND, entityName, file.getPath()); // NOSONAR
			}

			String headerLine = br.readLine();
			if(StringUtils.isEmpty(headerLine)) {
				throw new ErrorStatusException(CdsErrorStatuses.INVALID_CSV_FILE_NOHEADER, file.getPath()); // NOSONAR
			}
			if ( headerLine.indexOf(',') >= 0 ) { // we presume header titles don't have chars like "," or ";"
				if (headerLine.indexOf(';') >= 0) {
					throw new ErrorStatusException(CdsErrorStatuses.INVALID_CSV_FILE_INVALIDHEADER, file.getPath()); // NOSONAR
				}
				separator = ",";
			}

			headers = Arrays.stream(headerLine.split(separator)).map(n -> element(entity, n)).collect(Collectors.toList());
		}

		public CdsEntity entity() {
			return entity;
		}

		public Stream<Map<String, Object>> data() {
			if(headers.isEmpty()) {
				return Stream.empty();
			}
			return br.lines().map(this::data);
		}

		private Map<String, Object> data(String dataLine) {
			Map<String, Object> row = new HashMap<>();
			String[] values = dataLine.split(separator);
			for (int i = 0; i < headers.size(); i++) {
				CdsElement element = headers.get(i);
				String value = i >= values.length ? "": values[i];
				if ("".equals(value)) {
					continue;
				}
				if (element.getType().isSimple()) {
					try {
						row.put(element.getName(), CdsTypeUtils.parse(element.getType().as(CdsSimpleType.class).getType(), value));
					} catch(Exception e) {
						throw new ErrorStatusException(CdsErrorStatuses.INVALID_CSV_FILE_TYPEMISMATCH, value, element.getType().getName(), element.getName(), file.getPath(), e); // NOSONAR
					}
				}
			}

			return row;
		}

		private String entityName(String fileName) {
			return fileName.replace("-", ".").substring(0, fileName.length() - 4);
		}

		private CdsElement element(CdsEntity entity, String element) {
			return entity.elements().filter(e -> element.equalsIgnoreCase(e.getName())).findFirst()
					.orElseThrow(() -> new ErrorStatusException(CdsErrorStatuses.INVALID_CSV_FILE_UNKNOWNCOLUMN, file.getPath(), element, entity.getQualifiedName()));
		}

		@Override
		public void close() throws IOException {
			if (br != null) br.close();
		}
	}


}
