/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.request;

import java.util.Stack;
import java.util.concurrent.atomic.AtomicInteger;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.ServiceCatalog;
import com.sap.cds.services.messages.Messages;
import com.sap.cds.services.request.ParameterInfo;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.request.UserInfo;
import com.sap.cds.services.runtime.CdsRuntime;

public class RequestContextImpl implements RequestContextSPI {

	private final static Logger logger = LoggerFactory.getLogger(RequestContextImpl.class);

	private static AtomicInteger idProvider = new AtomicInteger();

	private static final ThreadLocal<Stack<RequestContextImpl>> requestContexts = ThreadLocal.withInitial(() -> new Stack<>());

	// properties of a RequestContext:
	private final CdsModel cdsModel;

	private final ServiceCatalog serviceCatalog;

	private final UserInfo userInfo;

	private final ParameterInfo parameterInfo;

	private final Messages messages;

	private final int id;

	private boolean isClosed;

	private RequestContextImpl(CdsModel cdsModel, ServiceCatalog serviceCatalog, UserInfo userInfo, ParameterInfo parameterInfo, Messages messages) {

		this.id = idProvider.incrementAndGet();

		this.serviceCatalog = serviceCatalog;
		this.cdsModel = cdsModel;
		this.userInfo = userInfo;
		this.parameterInfo = parameterInfo;
		this.messages = messages;
	}

	public static RequestContextSPI open(CdsModel cdsModel, ServiceCatalog serviceCatalog, UserInfo userInfo, ParameterInfo parameterInfo, Messages messages) {

		RequestContextImpl requestContext = new RequestContextImpl(cdsModel, serviceCatalog, userInfo, parameterInfo, messages);
		requestContexts.get().push(requestContext);

		logger.debug("Opened RequestContext {}", requestContext.getId());

		return requestContext;
	}

	/**
	 * Gives access to either the current {@link RequestContext},
	 * or if this does not exist to a {@link RequestContextHelper} that itself ensures that there is always a {@link RequestContext} opened.
	 * This method will throw a {@link NullPointerException}, if the passed {@link CdsRuntime} was null, but guarantees that it will never return <code>null</code>
	 *
	 * @param cdsRuntime A reference to the underlying {@link CdsRuntime} instance.
	 * @return the current {@link RequestContext} or the {@link RequestContextHelper}
	 */
	public static @Nonnull RequestContext getCurrentOrDefault(@Nonnull CdsRuntime cdsRuntime) {
		RequestContext current = getCurrentInternal();
		if(current == null) {
			current = new RequestContextHelper(cdsRuntime);
		}
		return current;
	}

	/**
	 * Gives access to either the current {@link RequestContext},
	 * or if this does not exist to a {@link RequestContextHelper} that itself ensures that there is always a {@link RequestContext} opened.
	 * THis method might return null, if the passed {@link CdsRuntime} was null.
	 *
	 * @param cdsRuntime A reference to the underlying {@link CdsRuntime} instance.
	 * @return the current {@link RequestContext}, the {@link RequestContextHelper} or <code>null</code>
	 */
	public static @Nullable RequestContext getCurrentOrNull(@Nullable CdsRuntime cdsRuntime) {
		RequestContext current = getCurrentInternal();
		if(current == null && cdsRuntime != null) {
			current = new RequestContextHelper(cdsRuntime);
		}
		return current;
	}

	/**
	 * Gives direct access to the current {@link RequestContext} stored in the {@link ThreadLocal}
	 * This may return <code>null</code> if there is no {@link RequestContext} currently opened
	 * @return the {@link RequestContext} stored in the {@link ThreadLocal}
	 */
	public static @Nullable RequestContextSPI getCurrentInternal() {
		Stack<RequestContextImpl> stack = requestContexts.get();
		return stack.isEmpty() ? null : stack.peek();
	}

	@Override
	public int getId() {
		return id;
	}

	@Override
	public CdsModel getModel() {
		return cdsModel;
	}

	@Override
	public ServiceCatalog getServiceCatalog() {
		return serviceCatalog;
	}

	@Override
	public UserInfo getUserInfo() {
		return userInfo;
	}

	@Override
	public ParameterInfo getParameterInfo() {
		return parameterInfo;
	}

	@Override
	public Messages getMessages() {
		return messages;
	}

	@Override
	public boolean isClosed() {
		return isClosed;
	}

	@Override
	public void close() {

		if (isClosed) {
			return;
		}

		isClosed = true;
		requestContexts.get().pop();
		logger.debug("Closed RequestContext {}", getId());
	}

}
