/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.utils;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.lang.reflect.WildcardType;

import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;

public class ReflectionUtils {

	private ReflectionUtils() {
		// hidden
	}

	/**
	 * Returns true if the the type is a {@link ParameterizedType}, using the upper bound in case of a {@link WildcardType}
	 * @param type the type to check
	 * @return true, if the type is a {@link ParameterizedType}
	 */
	public static boolean isParameterizedType(Type type) {
		if(type instanceof ParameterizedType) {
			return true;
		} else if (type instanceof WildcardType) {
			return isParameterizedType(((WildcardType) type).getUpperBounds()[0]);
		}
		return false;
	}

	/**
	 * Returns the type as {@link ParameterizedType}, using the upper bound in case of a {@link WildcardType}
	 * @param type the type to cast
	 * @return the casted type
	 */
	public static ParameterizedType getParameterizedType(Type type) {
		if(type instanceof ParameterizedType) {
			return (ParameterizedType) type;
		} else if (type instanceof WildcardType) {
			return getParameterizedType(((WildcardType) type).getUpperBounds()[0]);
		}
		throw new ErrorStatusException(CdsErrorStatuses.INVALID_TYPE, type.getTypeName());
	}

	/**
	 * Returns the class of a type, using the upper bound in case of a {@link WildcardType}
	 * @param type the type to resolve the class for
	 * @return the class of the type
	 * @throws ClassNotFoundException in case the class is not available
	 */
	public static Class<?> getClassForType(Type type) throws ClassNotFoundException {
		if (type instanceof WildcardType) {
			return getClassForType(((WildcardType) type).getUpperBounds()[0]);
		}
		return Class.forName(type.getTypeName());
	}

}
