/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.cds;

import java.util.List;
import java.util.Map;

import com.sap.cds.impl.parser.builder.SortSpecBuilder;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.StructuredTypeRef;
import com.sap.cds.ql.cqn.CqnLimit;
import com.sap.cds.ql.cqn.CqnModifier;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnSelectListItem;
import com.sap.cds.ql.cqn.CqnSortSpecification;
import com.sap.cds.ql.impl.ExpressionVisitor;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.cds.CdsReadEventContext;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.utils.TargetAwareCqnModifier;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.model.CdsAnnotations;

@ServiceName(value = "*", type = ApplicationService.class)
public class ImplicitSortingHandler implements EventHandler {

	@Before
	@HandlerOrder(OrderConstants.Before.FILTER_FIELDS)
	public void implicitSorting(CdsReadEventContext context) {
		CqnSelect select = context.getCqn();
		if (select.from().isRef() && select.groupBy().isEmpty()) {
			CqnSelect copy = CQL.copy(select, new OrderByModifier(context.getTarget()));

			context.setCqn(copy);
		}
	}

	public static class OrderByModifier extends TargetAwareCqnModifier {

		private boolean mainItemsContainFunctions = false;

		public OrderByModifier(CdsEntity target) {
			super(target);
		}

		@Override
		protected TargetAwareCqnModifier create(CdsEntity target) {
			return new OrderByModifier(target);
		}

		@Override
		public List<CqnSelectListItem> items(List<CqnSelectListItem> items) {
			mainItemsContainFunctions = containsFunctions(items);
			return super.items(items);
		}

		@Override
		public List<CqnSortSpecification> orderBy(List<CqnSortSpecification> orderBy) {
			if(!mainItemsContainFunctions) {
				CdsEntity target = getTarget();
				extendOrderBy(target, orderBy);
			}
			return super.orderBy(orderBy);
		}

		@Override
		public CqnSelectListItem expand(StructuredTypeRef ref, List<CqnSelectListItem> items, List<CqnSortSpecification> orderBy, CqnLimit limit) {
			CdsEntity refTarget = getRefTarget(ref);
			if(refTarget != null && !containsFunctions(items)) {
				extendOrderBy(refTarget, orderBy);
			}
			return super.expand(ref, items, orderBy, limit);
		}

		private void extendOrderBy(CdsEntity target, List<CqnSortSpecification> orderBy) {
			// ordering defined in the @cds.default.order annotation
			Object defaultOrder = CdsAnnotations.DEFAULT_ORDER.getOrDefault(target);
			if(defaultOrder != null) {
				if(defaultOrder instanceof List<?>) {
					((List<?>) defaultOrder).forEach(o -> handleDefaultOrderBy(o, orderBy));
				} else {
					handleDefaultOrderBy(defaultOrder, orderBy);
				}
			}

			// ordering defined in the underlying view
			// if we find an empty 'order by' on the underlying view we don't analyze the view's view
			target.query().ifPresent(view -> view.orderBy().stream()
					.filter(o -> !contains(o.item().displayName(), orderBy))
					.map(o -> ExpressionVisitor.copy(o, new CqnModifier(){}))
					.forEach(orderBy::add));

			// default ordering by keys
			// TODO: Should we ensure a stable order of the keys?
			target.keyElements()
			.filter(e -> e.getType().isSimple())
			.filter(e -> !contains(e.getName(), orderBy))
			.map(e -> CQL.get(e.getName()))
			.map(e -> SortSpecBuilder.by(e).build())
			.forEach(orderBy::add);
		}

		private void handleDefaultOrderBy(Object value, List<CqnSortSpecification> orderBy) {
			if (value instanceof Map<?, ?>) {
				Map<?, ?> map = (Map<?, ?>) value;
				Object by = map.get("by");
				Object element = map.get("=");
				if(by instanceof Map<?, ?>) {
					Object innerElement = ((Map<?, ?>) by).get("=");
					if(innerElement instanceof String && !contains((String) innerElement, orderBy)) {
						SortSpecBuilder builder = SortSpecBuilder.by(CQL.get((String) innerElement));
						Object desc = map.get("desc");
						Object asc = map.get("asc");
						if(Boolean.TRUE.equals(desc)) {
							builder = builder.desc();
						} else if(Boolean.TRUE.equals(asc)) {
							builder = builder.asc();
						}
						orderBy.add(builder.build());
					}
				} else if (element instanceof String && !contains((String) element, orderBy)) {
					orderBy.add(SortSpecBuilder.by(CQL.get((String) element)).build());
				}
			}
		}

		private boolean contains(String element, List<CqnSortSpecification> orderBy) {
			return orderBy.stream().anyMatch(c -> c.item().displayName().equals(element));
		}

		private boolean containsFunctions(List<CqnSelectListItem> items) {
			return items.stream().anyMatch(item -> item.isValue() && item.asValue().value().isFunction());
		}

	}

}
