/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.cds;

import static com.sap.cds.services.utils.model.CqnUtils.andPredicate;
import static com.sap.cds.services.utils.model.CqnUtils.modifiedWhere;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.ql.CQL;
import com.sap.cds.ql.cqn.CqnDelete;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnStatement;
import com.sap.cds.ql.cqn.CqnUpdate;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.authorization.AuthorizationService;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.cds.CdsDeleteEventContext;
import com.sap.cds.services.cds.CdsReadEventContext;
import com.sap.cds.services.cds.CdsUpdateEventContext;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.draft.DraftEditEventContext;
import com.sap.cds.services.draft.DraftSaveEventContext;
import com.sap.cds.services.draft.DraftService;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.utils.CdsModelUtils;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.DraftUtils;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;

@ServiceName(value = "*", type = ApplicationService.class)
public class AuthorizationHandler implements EventHandler {

	private final static Logger logger = LoggerFactory.getLogger(AuthorizationHandler.class);

	// Restriction annotations from underlying projected entities are added to the service entity by the compiler provided that no restrictions are defined already.
	// So the authorization has to be tested only on service level.
	@Before(event = "*")
	@HandlerOrder(OrderConstants.Before.CHECK_AUTHORIZATION)
	private void checkAuthorization(EventContext context) {
		// privileged users just skip authorization checks
		if ( !context.getUserInfo().isPrivileged() ) {
			authorizeOrThrow(context);
		}
	}

	private void authorizeOrThrow(EventContext context) {
		AuthorizationService authService = context.getServiceCatalog().getService(AuthorizationService.class, AuthorizationService.DEFAULT_NAME);

		final String event = context.getEvent();
		final String serviceName = ((ApplicationService) context.getService()).getDefinition().getQualifiedName();

		// need to check the service in all cases
		if (!authService.hasServiceAccess(serviceName, event)) {
			throw new ErrorStatusException(CdsErrorStatuses.EVENT_FORBIDDEN, event, serviceName);
		}

		// in case there is a target entity, check also the entity.
		// Note that this might be also true for bound actions and functions
		CdsEntity targetEntity = context.getTarget();
		String entityName = null;
		if (targetEntity != null) {
			entityName = targetEntity.getQualifiedName();

			CqnStatement query = null;
			Object cqn = context.get("cqn"); // TODO better work with new interface like CdsEventContext
			if (cqn instanceof CqnStatement) {
				query = (CqnStatement)cqn;
			}

			if (!authService.hasEntityAccess(entityName, event, query)) {
				throw new ErrorStatusException(CdsErrorStatuses.EVENT_FORBIDDEN, event, entityName);
			}
		}

		String actionOrFuntionName = (targetEntity == null ? (serviceName + "." + event) : event);

		// check actions or functions
		// Note that 'event' is the function/action name and depending on entityName are bound or unbound.
		// Note: isStandardCdsEvent() is not true for actions and functions
		if ( !CdsModelUtils.isStandardCdsEvent(event) ) {
			if ( CdsModelUtils.getAction(context.getModel(), entityName, actionOrFuntionName).isPresent() &&
					!authService.hasActionAccess(entityName, actionOrFuntionName)) {
				throw new ErrorStatusException(CdsErrorStatuses.EVENT_FORBIDDEN, actionOrFuntionName, entityName != null ? entityName : serviceName);

			} else if ( CdsModelUtils.getFunction(context.getModel(), entityName, actionOrFuntionName).isPresent() &&
					!authService.hasFunctionAccess(entityName, actionOrFuntionName)) {
				throw new ErrorStatusException(CdsErrorStatuses.EVENT_FORBIDDEN, actionOrFuntionName, entityName != null ? entityName : serviceName);
			}
		}

		// process instance-based authorization for some standard CDS events by appending the CQN where statements with additional restrictions
		if (context.getCdsRuntime().getEnvironment().getCdsProperties().getSecurity().getInstanceBasedAuthorization().isEnabled()) {
			if (event.equals(CqnService.EVENT_READ) ||
					event.equals(CqnService.EVENT_DELETE) ||
					event.equals(CqnService.EVENT_UPDATE) ||
					event.equals(DraftService.EVENT_DRAFT_EDIT) ||
					event.equals(DraftService.EVENT_DRAFT_SAVE)) {

				// finally handle where predicates for some standard cds events
				CqnPredicate wherePred = authService.calcWhereCondition(entityName, event);
				if (wherePred != null) {
					if (context.getEvent().equals(CqnService.EVENT_READ)) {
						CdsReadEventContext cdsContext = context.as(CdsReadEventContext.class);
						if (targetEntity != null && DraftUtils.isDraftEnabled(targetEntity)) {
							// omit wherePred for draft entities. wherePred still applies for active entities.
							wherePred = CQL.or(wherePred, CQL.get(Drafts.IS_ACTIVE_ENTITY).eq(false));
						}
						CqnSelect cqn = modifiedWhere(cdsContext.getCqn(), andPredicate(wherePred));
						cdsContext.setCqn(cqn);

					} else if (context.getEvent().equals(CqnService.EVENT_DELETE)) {
						CdsDeleteEventContext deleteContext = context.as(CdsDeleteEventContext.class);
						if (targetEntity != null && DraftUtils.isDraftEnabled(targetEntity)) {
							// omit wherePred for draft entities. wherePred still applies for active entities.
							wherePred = CQL.or(wherePred, CQL.get(Drafts.IS_ACTIVE_ENTITY).eq(false));
						}
						CqnDelete cqn = modifiedWhere(deleteContext.getCqn(), andPredicate(wherePred));
						deleteContext.setCqn(cqn);

					} else if (context.getEvent().equals(CqnService.EVENT_UPDATE)) {
						CdsUpdateEventContext updateContext = context.as(CdsUpdateEventContext.class);
						if (targetEntity != null && DraftUtils.isDraftEnabled(targetEntity)) {
							// omit wherePred for draft entities. wherePred still applies for active entities.
							wherePred = CQL.or(wherePred, CQL.get(Drafts.IS_ACTIVE_ENTITY).eq(false));
						}
						CqnUpdate cqn = modifiedWhere(updateContext.getCqn(), andPredicate(wherePred));
						updateContext.setCqn(cqn);

					} else if (event.equals(DraftService.EVENT_DRAFT_EDIT)) {
						DraftEditEventContext draftEditContext = context.as(DraftEditEventContext.class);
						CqnSelect cqn = modifiedWhere(draftEditContext.getCqn(), andPredicate(wherePred));
						draftEditContext.setCqn(cqn);

					} else if (event.equals(DraftService.EVENT_DRAFT_SAVE)) {
						DraftSaveEventContext saveContext = context.as(DraftSaveEventContext.class);
						CqnSelect cqn = modifiedWhere(saveContext.getCqn(), andPredicate(wherePred));
						saveContext.setCqn(cqn);
					}

					logger.debug("Created CQN authorization predicate {} for entity '{}' of service '{}', event {} and user {}",
							wherePred.toJson(), entityName, serviceName, event, context.getUserInfo().getName());

				} else {
					logger.debug("No CQN authorization predicate required for entity '{}' of service '{}', event {} and user {}",
							entityName, serviceName, event, context.getUserInfo().getName());
				}
			} else {
				// TODO: bound action and functions send CqnSelect as value of "cqn" key in the context. This select specifies the entities the action or function works on.
			}
		}

		// all checks derived from the model passes here.
	}

}
