/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.cds;

import java.util.Map;
import java.util.function.Predicate;

import com.sap.cds.reflect.CdsElement;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.utils.ValidatorUtils;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.StringUtils;
import com.sap.cds.services.utils.model.CdsAnnotations;

/**
 * Handler class to check mandatory elements
 * Mandatory elements are expected to be sent by the client
 * If mandatory elements are string based, empty strings are considered invalid as well
 */
@ServiceName(value = "*", type = ApplicationService.class)
public class MandatoryHandler implements EventHandler {

	@Before(event = { CqnService.EVENT_CREATE, CqnService.EVENT_UPDATE, CqnService.EVENT_UPSERT })
	@HandlerOrder(OrderConstants.Before.VALIDATE_FIELDS)
	public void runCheck(EventContext context) {
		ValidatorUtils.runNullElementCheck(context, (obj) -> {
			if (obj == null) {
				return true;
			} else if (obj instanceof String) {
				return StringUtils.isEmpty((String) obj);
			}
			return false;
		}, isNullNotAllowed(), (entry, type) -> {
			String elementName = entry.getKey().getName()
					+ (entry.getValue() != null ? ("' / '" + entry.getValue().getName()) : "");
			String entityName = type.getName();

			ValidatorUtils.handleValidationError(context, CdsErrorStatuses.VALUE_REQUIRED, elementName, entityName);
		});
	}

	/**
	 * Function that checks whether the specified cds element is not allowed to be
	 * null.
	 * 
	 * @return true, if cds element is not allowed to be null
	 */
	protected Predicate<CdsElement> isNullNotAllowed() {
		return (cdsElement) -> {
			boolean mandatory = CdsAnnotations.MANDATORY.isTrue(cdsElement);
			Map<?, ?> commonFieldControl = CdsAnnotations.COMMON_FIELDCONTROL.getOrDefault(cdsElement);
			return mandatory || (commonFieldControl != null && "Mandatory".equals(commonFieldControl.get("#")));
		};
	}

}
