/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.persistence;

import java.util.function.Predicate;

import com.sap.cds.reflect.CdsElement;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.utils.ValidatorUtils;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.model.CdsAnnotations;

/**
 * Handler class to check not null elements.
 * Adding not null to an element leads to a database constraint check.
 * This handler is mainly there to improve the error message in case such an element is set to null
 */
@ServiceName(value = "*", type = PersistenceService.class)
public class NotNullHandler implements EventHandler {

	@Before(event = { CqnService.EVENT_CREATE, CqnService.EVENT_UPDATE, CqnService.EVENT_UPSERT })
	@HandlerOrder(OrderConstants.Before.VALIDATE_FIELDS)
	public void runCheck(EventContext context) {
		ValidatorUtils.runNullElementCheck(context, (obj) -> obj == null, isNullNotAllowed(), (entry, type) -> {
			throw new ErrorStatusException(CdsErrorStatuses.VALUE_REQUIRED,
					entry.getKey().getName() + (entry.getValue() != null ? ("' / '" + entry.getValue().getName()) : ""),
					type.getName());
		});
	}

	/**
	 * Function that checks whether the specified cds element is not allowed to be
	 * null.
	 * 
	 * @return true, if cds element is not allowed to be null
	 */
	protected Predicate<CdsElement> isNullNotAllowed() {
		return (cdsElement) -> cdsElement.isNotNull() && CdsAnnotations.ASSERT_NOTNULL.isTrue(cdsElement);
	}

}
