package com.sap.cds.services.impl.utils;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.BiConsumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import org.slf4j.helpers.MessageFormatter;

import com.sap.cds.impl.DataProcessor;
import com.sap.cds.impl.DataProcessor.Action;
import com.sap.cds.ql.cqn.Path;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsSimpleType;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.services.ErrorStatus;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;

/**
 * Validation utilities.
 */
public class ValidatorUtils {

	/**
	 * Generic method that checks a given event context for null elements, utilizing
	 * the provided functions for the null element validation and the provided error
	 * handler in case of a failed validation.
	 * 
	 * @param context          the event context
	 * @param isNull           predicate that returns true if the provided object is
	 *                         null
	 * @param isNullNotAllowed predicate that returns true if the provided CDS
	 *                         element is not allowed to be null
	 * @param errorHandler     handles the error if validation fails
	 */
	public static void runNullElementCheck(EventContext context, Predicate<Object> isNull,
			Predicate<CdsElement> isNullNotAllowed,
			BiConsumer<Map.Entry<CdsElement, CdsElement>, CdsStructuredType> errorHandler) {

		boolean isPartialUpdate = CqnService.EVENT_UPDATE.equals(context.getEvent());

		DataProcessor.create().action(new Action() {
			@Override
			public void entries(Path path, CdsElement element, CdsStructuredType type,
					Iterable<Map<String, Object>> entries) {
				// key elements of the association target are not mandatory, as they get
				// auto-filled
				// returns an empty list, if parent is not set
				List<CdsElement> keyElements = CdsModelUtils.getAssociationKeys(element);

				// reduce association elements with two representations
				Map<CdsElement, CdsElement> reducedElements = CdsModelUtils.reduceAssociationElements(type);
				List<Map.Entry<CdsElement, CdsElement>> notNullElements = reducedElements.entrySet().stream()
						// filter all elements, that are keys of the parent association, they are
						// auto-filled by the database
						.filter(entry -> !(keyElements.contains(entry.getKey())
								|| keyElements.contains(entry.getValue())))
						// filter all not null elements, only the key is relevant here, as for the pair
						// always both elements have the same annotations
						.filter(entry -> isNullNotAllowed(entry.getKey(), isNullNotAllowed))
						.collect(Collectors.toList());

				if (notNullElements.isEmpty()) {
					return;
				}

				// check all not null element names
				for (Map<String, Object> map : entries) {
					for (Map.Entry<CdsElement, CdsElement> entry : notNullElements) {
						// if a partial update is allowed, first check if the key (or one of the pair)
						// is set in the data map
						if (isPartialUpdate && !map.containsKey(entry.getKey().getName())
								&& (entry.getValue() == null || !map.containsKey(entry.getValue().getName()))) {
							continue;
						}

						// if no pair is available only check the key in the map
						// if it is a pair only throw an exception if both elements are null
						if (isNull.test(map.get(entry.getKey().getName())) && (isNull.test(entry.getValue())
								|| isNull.test(map.get(entry.getValue().getName())))) {
							errorHandler.accept(entry, type);
						}
					}
				}
			}
		}).process(CdsServiceUtils.getEntitiesResolved(context), context.getTarget());
	}

	/**
	 * Returns true, if the element is not allowed to be <code>null</code>
	 * 
	 * @param element the element to validate
	 * @return true, if the element is not allowed to be <code>null</code>
	 */

	/**
	 * Checks whether the given cds element is not allowed to be null.
	 * 
	 * @param element          the element to validate
	 * @param isNullNotAllowed a predicate that returns true if the given cds
	 *                         element is not allowed to be null
	 * @return true, if the element is not allowed to be <code>null</code>
	 */
	public static boolean isNullNotAllowed(CdsElement element, Predicate<CdsElement> isNullNotAllowed) {
		if (element == null) {
			return false;
		}

		// elements with default values don't have to be checked
		if (element.getType().isSimple() && element.getType().as(CdsSimpleType.class).defaultValue().isPresent()) {
			return false;
		}

		if (element.getType().isAssociation() && element.getName().endsWith("_drafts")) {
			return false;
		}

		return isNullNotAllowed.test(element);
	}

	/**
	 * Handles the given {@link CdsErrorStatuses} by either generating an error
	 * message or throwing a {@link ErrorStatusException}.
	 * 
	 * @param context  the event context
	 * @param statuses the cds error status
	 * @param args     message args
	 */
	public static void handleValidationError(EventContext context, CdsErrorStatuses statuses, Object... args) {
		if (context.getCdsRuntime().getEnvironment().getCdsProperties().getErrors().isCombined()) {
			context.getMessages().error(getLocalizedMessage(context, statuses, args)).code(statuses.getCodeString());
		} else {
			throw new ErrorStatusException(statuses, args);
		}
	}

	/*
	 * Returns the localized message of an {@link ErrorStatus}.
	 * 
	 * @param context the event context
	 * 
	 * @param errorStatus the error status
	 * 
	 * @param args message args
	 * 
	 * @return localized error status message
	 */
	private static String getLocalizedMessage(EventContext context, ErrorStatus errorStatus, Object... args) {
		String localized = context.getCdsRuntime().getLocalizedMessage(errorStatus.getCodeString(), args,
				context.getParameterInfo().getLocale());

		if (Objects.equals(localized, errorStatus.getCodeString())) {
			return MessageFormatter.arrayFormat(errorStatus.getDescription(), args).getMessage();
		}
		return localized;
	}

}
