/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.cds;

import java.util.List;
import java.util.function.Predicate;
import java.util.stream.Stream;

import com.sap.cds.reflect.CdsService;
import com.sap.cds.services.ServiceCatalog;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.cds.RemoteService;
import com.sap.cds.services.environment.CdsProperties.Application;
import com.sap.cds.services.environment.CdsProperties.Application.ApplicationServiceConfig;
import com.sap.cds.services.runtime.CdsRuntimeConfiguration;
import com.sap.cds.services.runtime.CdsRuntimeConfigurer;

public class ApplicationServiceConfiguration implements CdsRuntimeConfiguration {

	@Override
	public int order() {
		return 100; // after other CdsService-based configurations
	}

	/**
	 * Returns a stream of application service configurations.
	 * By default application service configs for each CDS model service are returned.
	 * The CDS model services which are looked at can be limited by the {@code servicePredicate},
	 *
	 * If the configuration for an application service is explicitly defined, it will be returned by this method.
	 * If for a given CDS model service no explicit application service configuration can be obtained, a default service is created.
	 * This default service is not created, if a {@link RemoteService} for this CDS model service exists already.
	 *
	 * Overall only those configurations are returned, for which no other service already exists with the same name.
	 *
	 * @param configurer the {@link CdsRuntimeConfigurer}
	 * @param servicePredicate the predicate to filter relevant CDS model services
	 * @return the configurations of application services, which should be created
	 */
	public static Stream<ApplicationServiceConfig> applicationServiceConfigs(CdsRuntimeConfigurer configurer, Predicate<? super CdsService> servicePredicate) {
		Application appConfig = configurer.getCdsRuntime().getEnvironment().getCdsProperties().getApplication();
		ServiceCatalog catalog = configurer.getCdsRuntime().getServiceCatalog();

		return configurer.getCdsRuntime().getCdsModel().services()
		.filter(servicePredicate)
		.map(service -> service.getQualifiedName())
		.flatMap(name -> {
			List<ApplicationServiceConfig> serviceConfigs = appConfig.getServicesByModel(name);
			if(serviceConfigs.isEmpty()) {
				ApplicationServiceConfig config = appConfig.getService(name);
				if(config.getModel() == null &&
				   (appConfig.getServices().values().contains(config) ||
				    !catalog.getServices(RemoteService.class).anyMatch(s -> s.getDefinition().getQualifiedName().equals(name)))) {
					// default service with name as model service is only created
					// if it is not bound to a different model explicitly and
					// if it is configured explicitly (no fallback configuration)
					// or if no other service for that model definition exists yet
					return Stream.of(config);
				} else {
					return Stream.empty();
				}
			} else {
				return serviceConfigs.stream();
			}
		})
		.filter(config -> configurer.getCdsRuntime().getServiceCatalog().getService(config.getName()) == null);
	}

	@Override
	public void services(CdsRuntimeConfigurer configurer) {
		applicationServiceConfigs(configurer, s -> true)
		.map(config -> new ApplicationServiceImpl(config, configurer.getCdsRuntime()))
		.forEach(configurer::service);
	}

	@Override
	public void eventHandlers(CdsRuntimeConfigurer configurer) {
		// also draft enabled services require these handlers
		if(configurer.getCdsRuntime().getServiceCatalog().getServices(ApplicationService.class).findAny().isPresent()) {
			configurer.eventHandler(new ApplicationDefaultOnHandler())
			.eventHandler(new AuthorizationHandler())
			.eventHandler(new CapabilitiesHandler())
			.eventHandler(new SingletonHandler())
			.eventHandler(new ReadOnlyHandler())
			.eventHandler(new MandatoryHandler())
			.eventHandler(new RangeAssertionHandler())
			.eventHandler(new FormatAssertionHandler(configurer.getCdsRuntime()))
			.eventHandler(new EnumAssertionHandler())
			.eventHandler(new ValidationErrorHandler());

			if (configurer.getCdsRuntime().getEnvironment().getCdsProperties().getQuery().getImplicitSorting().isEnabled()) {
				configurer.eventHandler(new ImplicitSortingHandler());
			}
		}
	}

}
