/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.messaging.file;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileSystems;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;


/**
 * Implementation of the internal file based messaging service for local environment tests.
 *
 */
public class FileBasedMessagingBroker {

	private static final Logger logger = LoggerFactory.getLogger(FileBasedMessagingBroker.class);

	private static Set<String> createdBrokers = new HashSet<>();

	private final File file;
	private final WatchService fileWatcher;

	private final Map<String, List<String>> subscriptions = new HashMap<String, List<String>>();
	private final Map<String, List<QueueListener>> listeners = new HashMap<String, List<QueueListener>>();

	private FileBasedMessagingBroker(String name, File exchangeFile) throws IOException {

		this.file = exchangeFile;
		this.fileWatcher = FileSystems.getDefault().newWatchService();

		Path dir = Paths.get(file.getParentFile().getAbsolutePath());
		dir.register(fileWatcher, StandardWatchEventKinds.ENTRY_MODIFY);

		new Thread(() ->  {
			WatchKey watchKey = null;
			while (true) {
				try {
					watchKey = fileWatcher.poll(10, TimeUnit.MINUTES);
					if(watchKey != null) {
						if(!watchKey.pollEvents().isEmpty()) {
							receivedRawEvent();
						}
						watchKey.reset();
					}
				} catch(InterruptedException e) {
					logger.debug("File watching timed out, restarting watcher.");
				}
			}
		}, "FileBasedBroker: " + name).start();
	}

	/**
	 * Installs the file watcher of the specified file.
	 *
	 * @param name service name
	 * @param exchangeFile file to be used for the message exchange
	 *
	 * @return the file based broker instance
	 * @throws IOException in case, the service is already connected or the file watching initialization fails
	 */
	public static FileBasedMessagingBroker connect(String name, File exchangeFile) throws IOException {

		if (createdBrokers.contains(exchangeFile.getAbsolutePath())) {
			throw new IOException(String.format("The file based messaging broker for '%s' was already created!", exchangeFile.getAbsolutePath()));
		}

		FileBasedMessagingBroker broker = new FileBasedMessagingBroker(name, exchangeFile);
		createdBrokers.add(exchangeFile.getAbsolutePath());

		return broker;
	}

	/**
	 * Get notified when the file contents changed.
	 */
	private void receivedRawEvent() {

		List<MessageListeners> listeners = new ArrayList<>();

		try (RandomAccessFile raFile = new RandomAccessFile(file, "rw")) {

			FileChannel channel = raFile.getChannel();
			FileLock lock = null;
			try {
				lock = obtainFileLock(60000, channel);
				MessageLine line;
				List<MessageLine> leftLines = new ArrayList<>();
				while ((line = readNextLine(raFile)) != null) {
					if (line.isValid()) {
						List<QueueListener> queListeners = getQueueListeners(line.getTopic());
						if (queListeners != null) {
							listeners.add(new MessageListeners(line, queListeners));
						} else {
							leftLines.add(line);
						}
					}
				}

				if (!listeners.isEmpty()) {
					// cleanup the file
					raFile.setLength(0);
					if (!leftLines.isEmpty()) {
						for (MessageLine l : leftLines) {
							raFile.write(l.toString().getBytes(StandardCharsets.UTF_8));
						}
					}
				}
			} finally {
				try {
					if (lock != null) {
						lock.release();
					}
				} finally {
					channel.close();
				}
			}
		} catch (Exception e) { // NOSONAR
			logger.error("Could not read events", e);
		} finally {
			// notify listeners if any available
			listeners.forEach(msg -> msg.getListeners().forEach(listener -> {
				try {
					listener.received(msg.getMessage().getData(), msg.getMessage().getTopic(), msg.getMessage().getId());
				} catch (Throwable th) { // NOSONAR
					logger.error("error in the handler implementation!", th);
				}
			}));
		}
	}


	/**
	 * Reads the next line of the file and returns <code>null</code> at the end.
	 *
	 * @param file
	 * @return
	 * @throws IOException
	 */
	private MessageLine readNextLine(RandomAccessFile file) throws IOException {
		String line = file.readLine();
		if (line != null) {
			return new MessageLine(line);
		}
		return null;
	}

	/**
	 * Returns the listeners for the specified topic.
	 *
	 * @param topic
	 * @return
	 */
	private List<QueueListener> getQueueListeners(String topic) {
		synchronized (subscriptions) {
			List<QueueListener> result = new ArrayList<>();

			subscriptions.forEach((tpc, queues) -> {

				// TODO: match the topic
				String pattern = getTopicRegexp(tpc);
				if (topic.matches(pattern)) {
					queues.forEach(queue -> {
						List<QueueListener> qListeners = listeners.get(queue);
						if (qListeners != null && !qListeners.isEmpty()) {
							result.addAll(listeners.get(queue));
						}
					});
				}
			});

			if (!result.isEmpty()) {
				return result;
			}
		}
		return null;
	}

	private String getTopicRegexp(String eventPattern) {

		Objects.requireNonNull(eventPattern, "the topic pattern must not be null");

		return eventPattern
				.trim()
				.replace("*","\\*")
				.replaceAll("/\\\\\\*\\\\\\*$", "/.*")
				.replaceAll("\\\\\\*\\s*/", "([^/]*)/")
				.replaceAll("\\\\\\*$", "([^/]*)")
				;
	}

	/**
	 * Emits a new message to the file.
	 *
	 * @param topic the topic the message should be send to
	 * @param message the event message to be sent
	 */
	public void emitMessage(String topic, String message) {
		try (RandomAccessFile raFile = new RandomAccessFile(file, "rw")) {

			FileChannel channel = raFile.getChannel();
			FileLock lock = null;
			try {
				lock =  obtainFileLock(60000, channel);
				MessageLine line = new MessageLine(topic, message);
				channel.write(ByteBuffer.wrap(line.toString().getBytes(StandardCharsets.UTF_8)), channel.size());
			} finally {
				try {
					if (lock != null) {
						lock.release();
					}
				} finally {
					channel.close();
				}
			}
		} catch (Exception e) { // NOSONAR
			throw new ErrorStatusException(CdsErrorStatuses.EVENT_EMITTING_FAILED, topic, e);
		}
	}

	/**
	 * Registers a listener to the specified queue.
	 *
	 * @param queue the queue to listen on
	 * @param listener the listener
	 */
	public void listenToQueue(String queue, QueueListener listener) {
		if (!listeners.containsKey(queue)) {
			listeners.put(queue, new ArrayList<QueueListener>());
		}

		List<QueueListener> lists = listeners.get(queue);

		if (!lists.contains(listener)) {
			lists.add(listener);
		}
	}

	/**
	 * Subscribes a topic to the specified queue.
	 *
	 * @param queue the queue to subscribe to the topic
	 * @param topic the topic to be subscribed on
	 */
	public void subscribeTopic(String queue, String topic) {
		synchronized (subscriptions) {
			if (!subscriptions.containsKey(topic)) {
				subscriptions.put(topic, new ArrayList<String>());
			}

			List<String> queues = subscriptions.get(topic);
			if (!queues.contains(queue)) {
				queues.add(queue);
			}
		}
	}


	private FileLock obtainFileLock(long timeout, FileChannel channel)
			throws IOException, InterruptedException, TimeoutException {

		Long quitTime = System.currentTimeMillis() + timeout;

		do {
			try {
				FileLock lock = channel.tryLock();

				if (lock != null) {
					return lock;
				} else {
					Thread.sleep(1000);
				}
			}	catch (OverlappingFileLockException e) { // NOSONAR
				Thread.sleep(1000);
			}
		} while (System.currentTimeMillis() < quitTime);

		throw new TimeoutException();
	}

	/**
	 * Helper interface for queue listeners
	 */
	public static interface QueueListener {

		/**
		 * Notifies when a new message is received.
		 *
		 * @param message the raw message as string
		 * @param topic topic the message was addressed to
		 * @param messageId unique message ID
		 */
		public void received(String message, String topic, String messageId);
	}

	/**
	 * Helper for the message listeners relation.
	 */
	private static class MessageListeners {

		MessageLine message;
		List<QueueListener> listeners;

		public MessageListeners(MessageLine message, List<QueueListener> listeners) {
			super();
			this.message = message;
			this.listeners = listeners;
		}

		public MessageLine getMessage() {
			return message;
		}

		public List<QueueListener> getListeners() {
			return listeners;
		}
	}

	/**
	 * Helper for the message line representation in the file.
	 */
	private static class MessageLine {

		private JSONObject data;
		private boolean valid;
		private boolean isCloudEvent;

		public MessageLine(String line) throws IOException {
			super();
			try {
				this.data = new JSONObject(line);
				this.valid = true;
				this.isCloudEvent = isCloudEvent(this.data);
			} catch (Throwable th) { // NOSONAR
				throw new IOException("Invalid line representation!", th);
			}
		}

		private MessageLine(String topic, String data) {

			// check whether JSON cloud event
			this.data = getCloudEvent(data);

			if (this.data == null) {
				this.data = new JSONObject();
				this.data.put("data", data);
				this.isCloudEvent = true;
			} else {
				this.isCloudEvent = false;
			}

			this.data.put("event", topic);
			if (!this.data.has("id")) {
				this.data.put("id", UUID.randomUUID().toString());
			}
		}

		private boolean isCloudEvent(JSONObject event) {
			return event.has("data") && event.has("type") && event.has("source") && event.has("specversion");
		}

		private JSONObject getCloudEvent(String event) {

			try {
				JSONObject cloudEvent = new JSONObject(event);
				if (isCloudEvent(cloudEvent)) {
					return cloudEvent;
				}
			} catch (Throwable th) { // NOSONAR
				// not used
			}
			return null;
		}

		public String getTopic() {
			return data.get("event").toString();
		}
		public String getId() {
			return data.get("id").toString();
		}
		public String getData() {
			return isCloudEvent ? data.toString() : data.get("data").toString();
		}
		public boolean isValid() {
			return this.valid;
		}

		@Override
		public String toString() {
			return data.toString() + System.lineSeparator();
		}
	}
}
