/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.runtime;

import java.util.Locale;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.Service;
import com.sap.cds.services.authentication.AuthenticationInfo;
import com.sap.cds.services.changeset.ChangeSetContext;
import com.sap.cds.services.environment.ApplicationInfo;
import com.sap.cds.services.environment.ApplicationInfoProvider;
import com.sap.cds.services.environment.CdsEnvironment;
import com.sap.cds.services.environment.CdsProperties;
import com.sap.cds.services.environment.PropertiesProvider;
import com.sap.cds.services.environment.ServiceBinding;
import com.sap.cds.services.environment.ServiceBindingProvider;
import com.sap.cds.services.impl.ServiceCatalogImpl;
import com.sap.cds.services.impl.ServiceCatalogSPI;
import com.sap.cds.services.impl.ServiceExceptionUtilsImpl;
import com.sap.cds.services.impl.ServiceSPI;
import com.sap.cds.services.impl.environment.DefaultApplicationInfoProvider;
import com.sap.cds.services.impl.environment.DefaultServiceBindingProvider;
import com.sap.cds.services.impl.environment.SimplePropertiesProvider;
import com.sap.cds.services.impl.handlerregistry.HandlerRegistryTools;
import com.sap.cds.services.impl.messages.SimpleLocalizedMessageProvider;
import com.sap.cds.services.impl.request.ParameterInfoImpl;
import com.sap.cds.services.impl.request.UserInfoImpl;
import com.sap.cds.services.impl.runtime.mockusers.MockedUserInfoProvider;
import com.sap.cds.services.impl.utils.CdsModelUtils;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.messages.LocalizedMessageProvider;
import com.sap.cds.services.request.FeatureTogglesInfo;
import com.sap.cds.services.request.ParameterInfo;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.request.UserInfo;
import com.sap.cds.services.runtime.AuthenticationInfoProvider;
import com.sap.cds.services.runtime.CdsModelProvider;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.runtime.ChangeSetContextRunner;
import com.sap.cds.services.runtime.FeatureTogglesInfoProvider;
import com.sap.cds.services.runtime.ParameterInfoProvider;
import com.sap.cds.services.runtime.Request;
import com.sap.cds.services.runtime.RequestContextRunner;
import com.sap.cds.services.runtime.UserInfoProvider;

/**
 * Bootstrap for setting up service backbone. This includes reading the CDS
 * model, creating corresponding services and registering default and custom
 * handlers.
 */
@SuppressWarnings("deprecation")
public class CdsRuntimeImpl implements CdsRuntime {

	private final static Logger logger = LoggerFactory.getLogger(CdsRuntimeImpl.class);
	private final static String basePackage = "com.sap.cds.";

	private final CdsEnvironment environment;
	private final ServiceCatalogImpl serviceCatalog = new ServiceCatalogImpl();
	private CdsModel cdsModel = CdsModelUtils.loadCdsModel(null); // empty model

	private ServiceBindingProvider serviceBindingProvider = new DefaultServiceBindingProvider(this);
	private ApplicationInfoProvider applicationInfoProvider = new DefaultApplicationInfoProvider(this);

	private CdsModelProvider cdsModelProvider = new DefaultModelProvider();
	private UserInfoProvider userInfoProvider = new MockedUserInfoProvider(this); // will be overriden by authentication features
	private AuthenticationInfoProvider authenticationInfoProvider = () -> null; // needs to be set by the app framework
	private ParameterInfoProvider parameterInfoProvider = () -> null; // needs to be set by app framework
	private LocalizedMessageProvider localizedMessageProvider = new SimpleLocalizedMessageProvider();
	private FeatureTogglesInfoProvider featureToggleProvider = (userInfo, parameterInfo) -> null;

	// only visible in this package
	CdsRuntimeImpl(PropertiesProvider propertiesProvider) {
		this.environment = new CdsEnvironmentImpl(propertiesProvider);
	}

	@Override
	public ServiceCatalogSPI getServiceCatalog() {
		return serviceCatalog;
	}

	@Override
	public CdsModel getCdsModel() {
		return cdsModel;
	}

	@Override
	public CdsModel getCdsModel(UserInfo userInfo, FeatureTogglesInfo featureTogglesInfo) {
		if(userInfo.getTenant() == null) {
			return getCdsModel();
		}

		return cdsModelProvider.get(userInfo, featureTogglesInfo);
	}

	@Override
	public CdsEnvironment getEnvironment() {
		return environment;
	}

	@Override
	public ParameterInfo getProvidedParameterInfo() {
		ParameterInfo parameterInfo = parameterInfoProvider.get();
		return parameterInfo != null ? parameterInfo : ParameterInfo.create();
	}

	@Override
	public UserInfo getProvidedUserInfo() {
		UserInfo userInfo = userInfoProvider.get();
		return userInfo != null ? userInfo : UserInfo.create();
	}

	@Override
	public AuthenticationInfo getProvidedAuthenticationInfo() {
		return authenticationInfoProvider.get();
	}

	@Override
	public FeatureTogglesInfo getFeatureTogglesInfo(UserInfo userInfo, ParameterInfo parameterInfo) {
		FeatureTogglesInfo featureTogglesInfo = featureToggleProvider.get(userInfo, parameterInfo);
		return featureTogglesInfo != null ? featureTogglesInfo : FeatureTogglesInfo.create();
	}

	@Override
	public String getLocalizedMessage(String code, Object[] args, Locale locale) {
		return localizedMessageProvider.get(code, args, locale);
	}

	@Override
	public <T> T runInRequestContext(Request request, Function<RequestContext, T> requestHandler) {
		RequestContextRunner runInRequestContext = requestContext();
		if (request != null && request.getRequestUser() != null) {
			runInRequestContext.user( new UserInfoImpl(request.getRequestUser()) );
		}
		if (request != null && request.getRequestParameters() != null) {
			runInRequestContext.parameters(new ParameterInfoImpl(request.getRequestParameters(), this));
		}
		return runInRequestContext.run(requestHandler);
	}

	@Override
	public RequestContextRunner requestContext() {
		return new RequestContextRunnerImpl(this);
	}

	@Override
	public <T> T runInChangeSetContext(Function<ChangeSetContext, T> changeSetHandler) {
		return changeSetContext().run(changeSetHandler);
	}

	@Override
	public ChangeSetContextRunner changeSetContext() {
		return new ChangeSetContextRunnerImpl();
	}

	// These methods are made available through CdsRuntimeConfigurer
	// They are therefore NOT public, but only visible in this package

	void setCdsModel(String csnPath) {
		this.cdsModel = CdsModelUtils.loadCdsModel(csnPath);
	}

	void setCdsModel(CdsModel model) {
		this.cdsModel = model;
	}

	void setCdsModelProvider(CdsModelProvider provider) {
		cdsModelProvider = provider;
	}

	CdsModelProvider getCdsModelProvider() {
		return cdsModelProvider;
	}

	void setServiceBindingProvider(ServiceBindingProvider serviceBindingProvider) {
		this.serviceBindingProvider = serviceBindingProvider;
	}

	ServiceBindingProvider getServiceBindingProvider() {
		return serviceBindingProvider;
	}

	void setApplicationInfoProvider(ApplicationInfoProvider provider) {
		this.applicationInfoProvider = provider;
	}

	ApplicationInfoProvider getApplicationInfoProvider() {
		return applicationInfoProvider;
	}

	void setUserInfoProvider(UserInfoProvider userInfoProvider) {
		this.userInfoProvider = userInfoProvider;
	}

	UserInfoProvider getUserInfoProvider() {
		return userInfoProvider;
	}

	void setAuthenticationInfoProvider(AuthenticationInfoProvider authenticationInfoProvider) {
		this.authenticationInfoProvider = authenticationInfoProvider;
	}

	AuthenticationInfoProvider getAuthenticationInfoProvider() {
		return authenticationInfoProvider;
	}

	void setParameterInfoProvider(ParameterInfoProvider parameterInfoProvider) {
		this.parameterInfoProvider = parameterInfoProvider;
	}

	ParameterInfoProvider getParameterInfoProvider() {
		return parameterInfoProvider;
	}

	FeatureTogglesInfoProvider getFeatureToggleProvider() {
		return featureToggleProvider;
	}

	void setFeatureToggleProvider(FeatureTogglesInfoProvider featureToggleProvider) {
		this.featureToggleProvider = featureToggleProvider;
	}

	LocalizedMessageProvider getLocalizedMessageProvider() {
		return localizedMessageProvider;
	}

	void setLocalizedMessageProvider(LocalizedMessageProvider localizedMessageProvider) {
		this.localizedMessageProvider = localizedMessageProvider;
		ServiceExceptionUtilsImpl.defaultLocalizedMessageProvider = localizedMessageProvider;
	}

	void registerService(Service service) {
		serviceCatalog.register(service);
		ServiceSPI serviceSPI = CdsServiceUtils.getServiceSPI(service);
		if(serviceSPI != null) {
			serviceSPI.setCdsRuntime(this);
		}
	}

	<T> void registerEventHandler(Class<T> handlerClass, Supplier<T> handlerFactory) {
		HandlerRegistryTools.registerClass(handlerClass, handlerFactory, serviceCatalog);
		if(handlerClass.getPackage().getName().startsWith(basePackage)) {
			logger.debug("Registered handler class {}", handlerClass.getName());
		} else {
			logger.info("Registered handler class {}", handlerClass.getName());
		}
	}

	private class DefaultModelProvider implements CdsModelProvider {
		@Override
		public CdsModel get(UserInfo userInfo, FeatureTogglesInfo featureTogglesInfo) {
			return cdsModel;
		}
	}

	private class CdsEnvironmentImpl implements CdsEnvironment {

		private final CdsProperties properties;
		private final PropertiesProvider propertiesProvider;

		public CdsEnvironmentImpl(PropertiesProvider propertiesProvider) {
			if(propertiesProvider == null) {
				this.propertiesProvider = new SimplePropertiesProvider();
			} else {
				this.propertiesProvider = propertiesProvider;
			}

			this.properties = this.propertiesProvider.bindPropertyClass("cds", CdsProperties.class);
			ServiceExceptionUtilsImpl.errorsProperties = this.properties.getErrors();
		}

		@Override
		public CdsProperties getCdsProperties() {
			return properties;
		}

		@Override
		public <T> T getProperty(String key, Class<T> asClazz, T defaultValue) {
			return propertiesProvider.getProperty(key, asClazz, defaultValue);
		}

		@Override
		public Stream<ServiceBinding> getServiceBindings() {
			return serviceBindingProvider.get();
		}

		@Override
		public ApplicationInfo getApplicationInfo() {
			return applicationInfoProvider.get();
		}

	}

}
