/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.runtime;

import static com.sap.cds.services.impl.request.ParameterInfoFactory.emptyParameterInfo;
import static com.sap.cds.services.impl.request.UserInfoFactory.anonymousUserInfo;
import static com.sap.cds.services.impl.request.UserInfoFactory.privilegedUserInfo;

import java.util.Objects;
import java.util.function.Consumer;
import java.util.function.Function;

import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.ServiceCatalog;
import com.sap.cds.services.impl.messages.MessagesImpl;
import com.sap.cds.services.impl.request.RequestContextImpl;
import com.sap.cds.services.impl.request.RequestContextSPI;
import com.sap.cds.services.messages.Messages;
import com.sap.cds.services.request.FeatureTogglesInfo;
import com.sap.cds.services.request.ModifiableParameterInfo;
import com.sap.cds.services.request.ModifiableUserInfo;
import com.sap.cds.services.request.ParameterInfo;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.request.UserInfo;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.runtime.RequestContextRunner;


public class RequestContextRunnerImpl implements RequestContextRunner {

	/**
	 * A reference to the active runtime. Should be <code>null</code> in test scope only.
	 */
	private final CdsRuntime cdsRuntime;
	private final RequestContextSPI currentRequestContext;

	private UserInfo userInfo;
	private ParameterInfo parameterInfo;
	private FeatureTogglesInfo featureTogglesInfo;
	private boolean clearMessages = false;

	public RequestContextRunnerImpl(CdsRuntime cdsRuntime) {
		this.cdsRuntime = Objects.requireNonNull(cdsRuntime, "cdsRuntime must not be null");
		this.currentRequestContext = RequestContextImpl.getCurrentInternal();

		// ensures that provider-dependant info objects are always initialized as part of the parent thread
		// this comes at the cost of being potentially unnecessary, in case UserInfo or ParameterInfo objects are directly passed to the runner
		// however the benefit of usability of the API outweighs these drawbacks
		if (currentRequestContext != null) {
			userInfo = currentRequestContext.getUserInfo();
			parameterInfo = currentRequestContext.getParameterInfo();
			featureTogglesInfo = currentRequestContext.getFeatureTogglesInfo();
		} else {
			userInfo = providedUserInfo();
			parameterInfo = providedParameterInfo();
			featureTogglesInfo = calculateFeatureTogglesInfo();
		}
		assert userInfo != null;
		assert parameterInfo != null;
		assert featureTogglesInfo != null;
	}

	@Override
	public RequestContextRunner user(UserInfo userInfo) {
		Objects.requireNonNull(userInfo, "userInfo must not be null");
		this.userInfo = userInfo;
		return this;
	}

	@Override
	public RequestContextRunner anonymousUser() {
		userInfo = anonymousUserInfo();
		return this;
	}

	@Override
	public RequestContextRunner providedUser() {
		userInfo = providedUserInfo();
		return this;
	}

	@Override
	public RequestContextRunner privilegedUser() {
		String tenant = userInfo.getTenant();
		userInfo = privilegedUserInfo();
		// automatically propagate tenant for privileged users
		modifyUser(u -> u.setTenant(tenant));
		return this;
	}

	@Override
	public RequestContextRunner modifyUser(Consumer<ModifiableUserInfo> contextUser) {
		UserInfo prevUserInfo = userInfo;

		ModifiableUserInfo modifiedUser = prevUserInfo.copy();
		if (contextUser != null) {
			contextUser.accept(modifiedUser);
		}
		userInfo = modifiedUser;
		return this;
	}

	@Override
	public RequestContextRunner parameters(ParameterInfo parameterInfo) {
		Objects.requireNonNull(parameterInfo, "parameterInfo must not be null");
		this.parameterInfo = parameterInfo;
		return this;
	}

	@Override
	public RequestContextRunner featureToggles(FeatureTogglesInfo featureTogglesInfo) {
		Objects.requireNonNull(featureTogglesInfo, "featureTogglesInfo must not be null");
		this.featureTogglesInfo = featureTogglesInfo;
		return this;
	}

	@Override
	public RequestContextRunner clearParameters() {
		parameterInfo = emptyParameterInfo();
		return this;
	}

	@Override
	public RequestContextRunner providedParameters() {
		parameterInfo = providedParameterInfo();
		return this;
	}

	@Override
	public RequestContextRunner modifyParameters(Consumer<ModifiableParameterInfo> contextParamters) {
		ParameterInfo prevParameterInfo = parameterInfo;

		ModifiableParameterInfo modifiedParameter = prevParameterInfo.copy();
		if (contextParamters != null) {
			contextParamters.accept(modifiedParameter);
		}
		parameterInfo = modifiedParameter;

		return this;
	}

	@Override
	public RequestContextRunner clearMessages() {
		clearMessages = true;
		return this;
	}

	@Override
	public void run(Consumer<RequestContext> requestHandler) {
		run( (requestContext) -> {
			requestHandler.accept(requestContext);
			return Void.TYPE;
		});
	}

	@Override
	public <T> T run(Function<RequestContext, T> requestHandler) {
		ServiceCatalog serviceCatalog = cdsRuntime.getServiceCatalog(); // could be exposed by a provider later
		CdsModel cdsModel = cdsRuntime.getCdsModel(userInfo, featureTogglesInfo);
		Messages messages;
		if (!clearMessages && currentRequestContext != null) {
			messages = currentRequestContext.getMessages();
		} else {
			messages = new MessagesImpl(cdsRuntime, parameterInfo.getLocale());
		}

		try (RequestContextSPI requestContext = RequestContextImpl.open(cdsModel, serviceCatalog, userInfo, parameterInfo, featureTogglesInfo, messages)) {
			return requestHandler.apply(requestContext);
		}
	}

	private UserInfo providedUserInfo() {
		if (cdsRuntime != null) {
			UserInfo result = cdsRuntime.getProvidedUserInfo();
			if (result != null) {
				return result;
			}
		}
		return anonymousUserInfo();
	}

	private ParameterInfo providedParameterInfo() {
		if (cdsRuntime != null) {
			ParameterInfo result = cdsRuntime.getProvidedParameterInfo();
			if (result != null) {
				return result;
			}
		}
		return emptyParameterInfo();
	}

	private FeatureTogglesInfo calculateFeatureTogglesInfo() {
		if (cdsRuntime != null) {
			FeatureTogglesInfo result = cdsRuntime.getFeatureTogglesInfo(userInfo, parameterInfo);
			if (result != null) {
				return result;
			}
		}
		return FeatureTogglesInfo.create();
	}

	@Override
	public RequestContextRunner recalculateFeatureToggles() {
		this.featureTogglesInfo = calculateFeatureTogglesInfo();
		return this;
	}

}
