/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.authorization;

import java.util.List;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.authorization.ActionAccessEventContext;
import com.sap.cds.services.authorization.AuthorizationService;
import com.sap.cds.services.authorization.CalcWhereConditionEventContext;
import com.sap.cds.services.authorization.EntityAccessEventContext;
import com.sap.cds.services.authorization.FunctionAccessEventContext;
import com.sap.cds.services.authorization.ServiceAccessEventContext;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.authorization.PredicateResolver.MultipleAttributeValuesNotSupportedException;
import com.sap.cds.services.impl.changeset.ChangeSetContextImpl;
import com.sap.cds.services.impl.draft.ParentEntityLookup;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.StringUtils;
import com.sap.cds.services.utils.TenantAwareCache;
import com.sap.cds.services.utils.model.Privilege;
import com.sap.cds.services.utils.model.Restriction;

@ServiceName(value = "*", type = AuthorizationService.class)
public class AuthorizationDefaultOnHandler implements EventHandler {

	private final static Logger logger = LoggerFactory.getLogger(AuthorizationDefaultOnHandler.class);

	private final TenantAwareCache<RestrictionLookup, CdsModel> restrictionLookupCache;
	private final TenantAwareCache<PredicateLookup, CdsModel> predicateLookupCache;
	private final TenantAwareCache<ParentEntityLookup, CdsModel> parentEntityLookups;

	AuthorizationDefaultOnHandler(CdsRuntime runtime) {
		restrictionLookupCache = TenantAwareCache.create(() -> new RestrictionLookup(), runtime);
		predicateLookupCache = TenantAwareCache.create(() -> new PredicateLookup(), runtime);
		parentEntityLookups = TenantAwareCache.create(() -> new ParentEntityLookup(RequestContext.getCurrent(runtime).getModel()), runtime);
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected void defaultHasServiceAccess(ServiceAccessEventContext context) {

		final String event = context.getAccessEventName();
		final String service = context.getAccessServiceName();

		boolean result = true; // as long as there is no restriction access is granted

		Restriction restriction = restrictionLookupCache.findOrCreate().retrieveServiceRestriction(context.getModel(), service);
		if (restriction != null) {
			result = RestrictionUtils.passesRestriction(restriction, context.getUserInfo(), event);
		}

		context.setResult(result);
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected void defaultHasEntityAccess(EntityAccessEventContext context) {

		boolean result;

		ChangeSetContextImpl changeSetContext = (ChangeSetContextImpl) context.getChangeSetContext();
		// TODO: currently if the ChangeSetContext can only have a member which is the transaction manager registered in JdbcPersistenceService.ensureTransaction()
		if (changeSetContext == null || !changeSetContext.hasChangeSetMember()) {
			// Open own temporary ChangeSetContext which we don't want to left open (custom handlers might trigger remote calls)
			result = context.getCdsRuntime().changeSetContext().run(c -> {
				return defaultHasEntityAccessImpl(context);
			});
		} else {
			result = defaultHasEntityAccessImpl(context);
		}

		context.setResult(result);
	}

	protected boolean defaultHasEntityAccessImpl(EntityAccessEventContext context) {

		final String event = context.getAccessEventName();
		final String targetEntityName = context.getAccessEntityName();

		AuthorizationEntity authEntity = AuthorizationEntity.resolve(context,
				(model, entity) -> restrictionLookupCache.findOrCreate().retrieveEntityRestriction(model, entity),
				(entity) -> parentEntityLookups.findOrCreate().lookupParent(entity));

		if (authEntity.getAuthorizationEntity() == null) {
			// no "authorization entity" found -> reject as direct access to auto-exposed entity not allowed
			logger.debug("No authorization entity found when sending event '{}' to entity '{}'", event, targetEntityName);
			return false;
		}

		final String authorizationEntityName = authEntity.getAuthorizationEntity().getQualifiedName();

		// check static access to authorization entity in any case (target or path)
		Restriction restriction = restrictionLookupCache.findOrCreate().retrieveEntityRestriction(context.getModel(), authorizationEntityName);
		if (restriction != null && !RestrictionUtils.passesRestriction(restriction, context.getUserInfo(), event)) {
			if (authorizationEntityName.equals(targetEntityName)) {
				logger.debug("No authorization to send event '{}' to entity '{}'",
						event, targetEntityName);
			} else {
				logger.debug("No authorization to send event '{}' to entity '{}' because of authorization entity '{}'",
						event, targetEntityName, authorizationEntityName);
			}
			return false;
		}

		// check instance-based condition on path if required
		// We can't test by sending READ on the service level as 'event' implies a different conditions in general.
		// Instead we take the condition of 'event' and test the visibility of the instance with the condition applied.
		// Note on drafts: the authorization is enforced by DraftHandler which will filter all draft documents with the correct owner.
		if ( !authorizationEntityName.equals(targetEntityName) && authEntity.getAuthorizationPath() != null && !authEntity.isDraft()) {
			CqnPredicate authorizationCondition = authEntity.authorizationCondition(authEntity.getAuthorizationEntity(), event);
			if (authorizationCondition != null) {
				CqnSelect readTest = Select.from(authEntity.getAuthorizationPath()).where(authorizationCondition);
				// TODO: This makes generic authorization handler dependent from persistence service, but sending to the
				// corresponding application server is not correct (generic and custom handlers would be called with wrong event 'READ').
				// We decided to tackle this general problem when we start to support several persistence service instances.
				if ( !CdsServiceUtils.getDefaultPersistenceService(context).run(readTest).first().isPresent() ) {
					logger.debug("No authorization to send event '{}' to entity '{}' because of instance '{}'",
							event, targetEntityName, authEntity.getAuthorizationPath());
					return false;
				}
			}
		}

		// no further restrictions
		return true;
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected void defaultHasFunctionAccess(FunctionAccessEventContext context) {

		final String functionName = context.getFunctionName();
		final String entityName = context.getEntityName();

		boolean result = true; // as long as there is no restriction access is granted

		Restriction restriction = restrictionLookupCache.findOrCreate().lookupFunctionRestriction(context.getModel(), entityName, functionName);
		if (restriction != null) {
			result = RestrictionUtils.passesRestriction(restriction, context.getUserInfo(), Privilege.PredefinedGrant.ALL.toString() /* = event: restrictions on functions have implicitly grant '*' */);
		}

		context.setResult(result);
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected void defaultHasActionAccess(ActionAccessEventContext context) {

		final String actionName = context.getActionName();
		final String entityName = context.getEntityName();

		boolean result = true; // as long as there is no restriction access is granted

		Restriction restriction = restrictionLookupCache.findOrCreate().lookupActionRestriction(context.getModel(), entityName, actionName);
		if (restriction != null) {
			result = RestrictionUtils.passesRestriction(restriction, context.getUserInfo(), Privilege.PredefinedGrant.ALL.toString() /* = event: restrictions on actions have implicitly grant '*' */);
		}

		context.setResult(result);
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected void defaultCalcWhereCondition(CalcWhereConditionEventContext context) {

		final String event = context.getEventName();
		final String entityName = context.getEntityName();

		CqnPredicate result = null; // no filter as long there is no where condition hit

		if ( !context.getUserInfo().isPrivileged() ) {
			// we can't add where clauses to CREATE / UPSERT events!
			Restriction restriction = restrictionLookupCache.findOrCreate().retrieveEntityRestriction(context.getModel(), entityName);
			if (restriction != null) {
				List<Privilege> privileges = RestrictionUtils.passingPrivilegesOfRestriction(restriction, context.getUserInfo(), event).collect(Collectors.toList());
				for (Privilege privilege : privileges) {
					CqnPredicate cqnPredicate = null;
					if ( !StringUtils.isEmpty(privilege.getCxnWhereCondition()) ) {
						try {
							cqnPredicate = predicateLookupCache.findOrCreate().resolvePredicate(privilege.getCxnWhereCondition(), context.getUserInfo());

						} catch(MultipleAttributeValuesNotSupportedException e) {
							logger.debug("No authorization to send event '{}' to entity '{}' because user {} has multiple values for attribute '{}' (filter resource '{}')",
									event, entityName, context.getUserInfo().getName(), e.getAttributeName(), e.getResourceName(), e);
							throw new ErrorStatusException(CdsErrorStatuses.EVENT_FORBIDDEN_UNSUPPORTED_USER_ATTRIBUTES, event, entityName, e.getAttributeName(), e);

						} catch (Exception e) { // NOSONAR
							throw new ErrorStatusException(CdsErrorStatuses.INVALID_WHERE_CONDITION, privilege.getCxnWhereCondition(), entityName, event, context.getUserInfo().getName(), e);
						}
					} else if( !StringUtils.isEmpty(privilege.getWhereCondition()) ) {
						// no _where property (Cxn) generated for where clause, e.g. compilation error silently ignored
						throw new ErrorStatusException(CdsErrorStatuses.INCONSISTENT_WHERE_CONDITION, privilege.getWhereCondition(), entityName);
					}

					if (cqnPredicate == null) { // null evaluates to 'true'
						result = null;
						break;
					}
					result = (result != null) ? CQL.or(result, cqnPredicate) : cqnPredicate;
				}
			}
		}

		context.setResult(result);
	}

}
