/*********************************************************************
 * (C) 2020 SAP SE or an SAP affiliate company. All rights reserved. *
 *********************************************************************/
package com.sap.cds.services.impl.authorization;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.function.Function;

import com.sap.cds.Result;
import com.sap.cds.impl.builder.model.Conjunction;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.cqn.AnalysisResult;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnStatement;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.ResolvedSegment;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.authorization.EntityAccessEventContext;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.draft.DraftService;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.impl.draft.ParentEntityLookup.ParentEntityLookupResult;
import com.sap.cds.services.impl.utils.CdsModelUtils;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.DraftUtils;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.model.CdsAnnotations;
import com.sap.cds.services.utils.model.Restriction;

class AuthorizationEntity {

	private final EntityAccessEventContext context;

	private final BiFunction<CdsModel, String, Restriction> restrictionLookup;

	private final Function<CdsEntity, List<ParentEntityLookupResult>> parentEntityLookup;

	private CdsEntity authorizationEntity;

	private CqnStructuredTypeRef authorizationPath;

	// indicates if a draft document is touched
	private boolean isDraft;

	public static AuthorizationEntity resolve(EntityAccessEventContext context,
			BiFunction<CdsModel, String, Restriction> restrictionLookup,
			Function<CdsEntity, List<ParentEntityLookupResult>> parentEntityLookup) {
		return new AuthorizationEntity(context, restrictionLookup, parentEntityLookup);
	}

	public CdsEntity getAuthorizationEntity() {
		return authorizationEntity;
	}

	public CqnStructuredTypeRef getAuthorizationPath() {
		return authorizationPath;
	}

	public boolean isDraft() {
		return isDraft;
	}

	public CqnPredicate authorizationCondition(CdsEntity authorizationEntity, String event) {

		if (context.getCdsRuntime().getEnvironment().getCdsProperties().getSecurity().getInstanceBasedAuthorization().isEnabled()) {
			if (event.equals(CqnService.EVENT_READ) ||
					event.equals(CqnService.EVENT_DELETE) ||
					event.equals(CqnService.EVENT_UPDATE) ||
					event.equals(DraftService.EVENT_DRAFT_EDIT) ||
					event.equals(DraftService.EVENT_DRAFT_SAVE)) { // TODO: refactor -> instance-based code
				return context.getService().calcWhereCondition(authorizationEntity.getQualifiedName(), event);
			}
		}
		return null;
	}

	private AuthorizationEntity(EntityAccessEventContext context,
			BiFunction<CdsModel, String, Restriction> restrictionLookup,
			Function<CdsEntity, List<ParentEntityLookupResult>> parentEntityLookup) {
		this.context = context;
		this.restrictionLookup = restrictionLookup;
		this.parentEntityLookup = parentEntityLookup;

		if (context.getAccessQuery() != null &&
				context.getCdsRuntime().getEnvironment().getCdsProperties().getSecurity().getAuthorizeAutoExposedEntities().isEnabled()) {
			findAuthorizationEntity(context.getAccessQuery());
		} else {
			authorizationEntity = context.getTarget(); // support the old path w/o query parameter or switched off explicitly
		}
	}

	private void findAuthorizationEntity(CqnStatement query) {

		List<Segment> tmpAuthorizationPath = new ArrayList<>();

		AnalysisResult pathAnalysis = CdsModelUtils.getEntityPath(query, context.getModel());
		for (Iterator<ResolvedSegment> segsRevIter = pathAnalysis.reverse(); segsRevIter.hasNext();) {
			ResolvedSegment segment = segsRevIter.next();
			if ( authorizationEntity == null && isAuthorizationEntity(segment.entity()) ) {
				authorizationEntity = segment.entity();
			}
			if (authorizationEntity != null) {
				// collect the path prefix including authorizationSegment
				tmpAuthorizationPath.add(0, segment.segment());
			}
			if (!isDraft && DraftUtils.isDraftEnabled(segment.entity()) && segment.keys() != null) {
				// check if we touch a draft document. At least one key element must be present in this case (e.g. root).
				Boolean isActiveEntity = (Boolean) segment.keys().get(Drafts.IS_ACTIVE_ENTITY);
				isDraft = Boolean.FALSE.equals(isActiveEntity);
			}
		}
		assert authorizationEntity == null || !tmpAuthorizationPath.isEmpty();

		if (authorizationEntity != null) {
			authorizationPath = CQL.to(tmpAuthorizationPath).asRef();

		} else if (DraftUtils.isDraftEnabled(pathAnalysis.targetEntity())) {
			// give draft-enabled a second chance: Fiori drafts require direct access, but path still can be constructed
			if (isDraft) {
				// shortcut: only current user is allowed to access (enforced by draft handler)
				this.authorizationEntity = pathAnalysis.targetEntity();
			} else {
				// Note: is necessarily not a draft entity, key elements must be present
				findAuthorizationEntity(pathAnalysis.targetEntity(), pathAnalysis.targetKeys());
			}
		}

		// else: direct access to auto-exposed entity (no derived authorization possible)
	}

	private boolean isAuthorizationEntity(CdsEntity entity) {

		// TODO: remove text entity workaround after compiler fix.
		// TODO: make text table test more reliable (e.g. entity names could accidently end with "_texts")

		// Special rule: A _text table is only an authorization entities, if and only if it bears a concrete restriction.
		// The rational behind this is that it is not possible for the runtime to distinguish between a generated resp. manually added table.
		boolean isTextTable = entity.getQualifiedName().endsWith("_texts") || entity.getQualifiedName().endsWith(".texts");

		// 1. explicit entity
		boolean autoexposed = CdsAnnotations.AUTOEXPOSED.getOrDefault(entity);
		if (!autoexposed && !isTextTable) {
			return true;
		}

		// 2. auto-expose (alias CodeList) entity
		boolean autoexpose = CdsAnnotations.AUTOEXPOSE.getOrDefault(entity);
		if (autoexposed && autoexpose && !isTextTable) {
			return true; // is implicitly @readonly as enforced by the CapabilityHandler
		}

		// 3. entity with explicit restrictions
		Restriction restriction = restrictionLookup.apply(context.getModel(), entity.getQualifiedName());
		if (restriction != null) {
			return true;
		}

		// this is an auto-exposed entity for which authorization needs to be derived
		return false;
	}

	private void findAuthorizationEntity(CdsEntity targetEntity, Map<String, Object> targetKeys) {

		// try to go the fast way on entities only
		for (CdsEntity testEntity = targetEntity; authorizationEntity == null;) {
			List<ParentEntityLookupResult> parentsLookup = parentEntityLookup.apply(testEntity);
			if (parentsLookup == null || parentsLookup.isEmpty()) {
				throw new ErrorStatusException(CdsErrorStatuses.NO_PARENT_ENTITY, testEntity.getQualifiedName());
			} else if (parentsLookup.size() > 1) {
				break;
			}
			CdsEntity entity = parentsLookup.get(0).getParentEntity();
			if (isAuthorizationEntity(entity) ) {
				authorizationEntity = entity;
			} else {
				testEntity = entity;
			}
		}

		// check if authorizationEntityKeys for instance-based auth are required
		if (authorizationEntity != null) {
			Restriction restriction = restrictionLookup.apply(context.getModel(), authorizationEntity.getQualifiedName());
			if (restriction != null && authorizationCondition(authorizationEntity, context.getAccessEventName()) != null) {
				authorizationEntity = null;
			}
		}

		if (authorizationEntity == null) {

			// go the expensive way: constructing the path of entity instances. Precondition: single target instance.
			// This is given according to OData spec/Fiori usage (PATCH or DELETE). All other events don't use direct access.
			// This will go away in future when OData containment is supported by Fiori and hence paths are used for all types of requests.

			if (targetKeys == null || targetKeys.isEmpty()) {
				return; // can't construct paths w/o target instance.
			}

			CdsEntity pathEntity = targetEntity;
			Map<String, Object> pathEntityKeys = targetKeys;
			while (authorizationEntity == null) {

				List<ParentEntityLookupResult> parentsLookup = parentEntityLookup.apply(pathEntity);
				if (parentsLookup == null || parentsLookup.isEmpty()) {
					throw new ErrorStatusException(CdsErrorStatuses.PARENT_NOT_EXISTING, pathEntity.getQualifiedName());
				}

				int numberParents = 0;
				CdsEntity parentPathEntity = null;
				Map<String, Object> parentPathEntityKeys = null;
				for (ParentEntityLookupResult parentLookup : parentsLookup) {
					String compositionName = parentLookup.getComposition().getName();
					CqnPredicate pred = pathEntityKeys.entrySet().stream()
							.map(key -> CQL.to(compositionName).get(key.getKey()).eq(key.getValue()))
							.collect(Conjunction.and());

					CqnSelect select = Select.from(parentLookup.getParentEntity()).where(pred);
					Result result = CdsServiceUtils.getDefaultPersistenceService(context).run(select);
					if (result.first().isPresent()) {
						parentPathEntity = parentLookup.getParentEntity();
						parentPathEntityKeys = extractKeys(result.first().get(), parentLookup.getParentEntity());
						numberParents += result.rowCount();
					}
				}
				if (parentPathEntity == null) {
					throw new ErrorStatusException(CdsErrorStatuses.PARENT_NOT_EXISTING, pathEntity.getQualifiedName());
				} else if (numberParents > 1) {
					throw new ErrorStatusException(CdsErrorStatuses.MULTIPLE_PARENTS, numberParents, pathEntity.getQualifiedName());
				}

				if (isAuthorizationEntity(parentPathEntity) ) {
					authorizationEntity = parentPathEntity;
					authorizationPath = CQL.entity(authorizationEntity.getQualifiedName()).filter(CQL.matching(parentPathEntityKeys)).asRef();
				} else {
					pathEntity = parentPathEntity;
					pathEntityKeys = parentPathEntityKeys;
				}
			}
		}
	}

	private static Map<String, Object> extractKeys(Map<String, Object> m, CdsEntity entity) {
		Map<String, Object> result = new HashMap<>(m);
		result.entrySet().removeIf(entry -> {
			return !entity.findElement(entry.getKey()).map(e -> e.isKey()).orElse(false)
					&& !Drafts.IS_ACTIVE_ENTITY.equals((entry.getKey()));

		});
		if (result.size() != entity.keyElements().count()) {
			throw new ErrorStatusException(CdsErrorStatuses.NO_KEYS_IN_RESULT, entity);
		}
		return result;
	}

}
