/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.cds;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Limit;
import com.sap.cds.ql.StructuredTypeRef;
import com.sap.cds.ql.cqn.CqnLimit;
import com.sap.cds.ql.cqn.CqnModifier;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnSelectListItem;
import com.sap.cds.ql.cqn.CqnSortSpecification;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.ql.impl.ExpressionVisitor;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.cds.CdsReadEventContext;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.utils.TargetAwareCqnModifier;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.model.CdsAnnotations;

@ServiceName(value = "*", type = ApplicationService.class)
public class ImplicitSortingHandler implements EventHandler {

	@Before
	@HandlerOrder(OrderConstants.Before.FILTER_FIELDS)
	public void implicitSorting(CdsReadEventContext context) {
		CqnSelect select = context.getCqn();
		if (select.from().isRef()) {
			CqnSelect copy = CQL.copy(select, new OrderByModifier(context.getTarget()));
			context.setCqn(copy);
		}
	}

	public static class OrderByModifier extends TargetAwareCqnModifier {

		private boolean mainOnlyFunctions = false;
		private List<CqnValue> mainGroupBy = Collections.emptyList();
		private boolean mainLimit = false;

		public OrderByModifier(CdsEntity target) {
			super(target);
		}

		@Override
		protected TargetAwareCqnModifier create(CdsEntity target) {
			return new OrderByModifier(target);
		}

		@Override
		public List<CqnSelectListItem> items(List<CqnSelectListItem> items) {
			mainOnlyFunctions = containsOnlyFunctions(items);
			return super.items(items);
		}

		@Override
		public List<CqnValue> groupBy(List<CqnValue> groupBy) {
			mainGroupBy = groupBy;
			return super.groupBy(groupBy);
		}

		@Override
		public CqnLimit limit(Limit limit) {
			mainLimit = limit != null;
			return super.limit(limit);
		}

		@Override
		public List<CqnSortSpecification> orderBy(List<CqnSortSpecification> orderBy) {
			return extendOrderBy(getTarget(), orderBy, mainOnlyFunctions, mainGroupBy, mainLimit);
		}

		@Override
		public CqnSelectListItem expand(StructuredTypeRef ref, List<CqnSelectListItem> items, List<CqnSortSpecification> orderBy, CqnLimit limit) {
			CdsEntity refTarget = getRefTarget(ref);
			if(refTarget != null) {
				orderBy = extendOrderBy(refTarget, orderBy, containsOnlyFunctions(items), Collections.emptyList(), limit != null);
			}
			return super.expand(ref, items, orderBy, limit);
		}

		private List<CqnSortSpecification> extendOrderBy(CdsEntity target, List<CqnSortSpecification> orderBy, boolean onlyFunctions, List<CqnValue> groupBy, boolean limit) {
			// in case of functions we assume the functions are aggregating and the result only contains a single row
			// therefore we don't add any sorting specifications
			// this might not always be the case, but for now we ignore these edge cases
			if (onlyFunctions) {
				return orderBy;
			}

			Set<String> sortItems = new HashSet<>();
			orderBy.stream().map(si -> si.value().toJson()).forEach(sortItems::add);

			// apply explicitly modeled sorting specifications
			// if query is structurally equivalent to the entity
			if (groupBy.isEmpty()) {
				// ordering defined in the @cds.default.order annotation
				Object defaultOrder = CdsAnnotations.DEFAULT_ORDER.getOrDefault(target);
				handleDefaultOrderBy(defaultOrder, orderBy);

				// ordering defined in the underlying view
				// if we find an empty 'order by' on the underlying view we don't analyze the view's view
				target.query().ifPresent(view -> view.orderBy().stream().forEach(o -> {
					String json = o.value().toJson();
					if (!sortItems.contains(json)) {
						sortItems.add(o.value().toJson());
						orderBy.add(ExpressionVisitor.copy(o, new CqnModifier() {
						}));
					}
				}));
			}

			// implicit sorting to ensure stable paging
			if (limit) {
				if (groupBy.isEmpty()) {
					// sort by keys
					// TODO: Should we ensure a stable order of the keys?
					target.keyElements()
							.filter(e -> e.getType().isSimple())
							.filter(e -> !contains(e.getName(), orderBy))
							.map(e -> CQL.get(e.getName()).asc())
							.forEach(orderBy::add);
				} else {
					// sort by all group by elements
					groupBy.stream()
							.filter(CqnValue::isRef)
							.map(v -> CQL.get(v.asRef().segments()).asc())
							.forEach(orderBy::add);
				}
			}
			return orderBy;
		}

		private void handleDefaultOrderBy(Object value, List<CqnSortSpecification> orderBy) {
			if (value instanceof Map<?, ?>) {
				Map<?, ?> map = (Map<?, ?>) value;
				Object by = map.get("by");
				Object element = map.get("=");
				if(by instanceof Map<?, ?>) {
					Object innerElement = ((Map<?, ?>) by).get("=");
					if(innerElement instanceof String && !contains((String) innerElement, orderBy)) {
						CqnValue val = CQL.get((String) innerElement);
						Object desc = map.get("desc");
						CqnSortSpecification.Order order = CqnSortSpecification.Order.ASC;
						if(Boolean.TRUE.equals(desc)) {
							order = CqnSortSpecification.Order.DESC;
						}
						orderBy.add(CQL.sort(val, order));
					}
				} else if (element instanceof String && !contains((String) element, orderBy)) {
					orderBy.add(CQL.get((String) element).asc());
				}
			} else if (value instanceof List<?>) {
				((List<?>) value).forEach(o -> handleDefaultOrderBy(o, orderBy));
			}
		}

		private boolean contains(String element, List<CqnSortSpecification> orderBy) {
			return orderBy.stream().anyMatch(c -> refNamed(c.value(), element));
		}

		private boolean refNamed(CqnValue val, String name) {
			if (!val.isRef()) {
				return false;
			}

			return name.equals(val.asRef().displayName());
		}

		private boolean containsOnlyFunctions(List<CqnSelectListItem> items) {
			return !items.isEmpty() && items.stream().allMatch(item -> item.isValue() && item.asValue().value().isFunction());
		}

	}

}


