/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.composite;

import java.util.Arrays;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.services.EventContext;
import com.sap.cds.services.Service;
import com.sap.cds.services.environment.CdsProperties.Composite.CompositeServiceConfig;
import com.sap.cds.services.handler.EventPredicate;
import com.sap.cds.services.handler.Handler;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.StringUtils;

/**
 * The composite service is responsible for dispatching events to configured services. The configuration
 * is also used for delegating the handler registration to the configured services.
 *
 * Example:
 *
 * <code>
 * name: my_composite
 * routes:
 *   - service: my_service
 *     events:
 *       - 'CREATE'
 *       - 'READ'
 * </code>
 *
 * As shown in the configuration example above, the events 'CREATE' and 'READ' are dispatched to
 * 'my_service' when they are emitted on 'my_composite'. Also if a handler registration
 * for these events is created on 'my_composite' the registration is delegated to 'my_service'.
 *
 */
public class CompositeService implements Service {

	private static final Logger logger = LoggerFactory.getLogger(CompositeService.class);

	private final String name;
	private final CdsRuntime runtime;
	private final Map<Pattern, Set<String>> eventMatcherToServices = new LinkedHashMap<>();

	public CompositeService(CompositeServiceConfig config, CdsRuntime runtime) {
		this.name = config.getName();
		this.runtime = runtime;

		// create the event matcher table in order to simplify the event to services lookup
		Map<String, Set<String>> eventToServices = new LinkedHashMap<>();

		config.getRoutes().forEach(route -> {
			route.getEvents().forEach(event -> {
				Set<String> services = eventToServices.get(event);
				if (services == null) {
					services = new HashSet<>();
					eventToServices.put(event, services);
				}

				if (!StringUtils.isEmpty(route.getService())) {
					services.add(route.getService());
				} else {
					throw new ErrorStatusException(CdsErrorStatuses.NO_DESTINATION_SERVICE, String.join(", ", route.getEvents()), getName());
				}
			});
		});

		// create the matcher table
		eventToServices.forEach((event, services) -> {
			eventMatcherToServices.put(CompositeUtils.getEventMatcherRegexp(event), services);
		});

		dump();
	}

	@Override
	public String getName() {
		return name;
	}

	@Override
	public void on(String[] events, String[] entities, int order, Handler handler) {
		for (String event: events) {
			Service destinationService = getDestinationService(event);
			logger.debug("Delegated the composite service '{}' ON handler registration for event '{}' to service '{}'", getName(), event, destinationService.getName());
			destinationService.on(new String[] {event}, entities, order, handler);
		}
	}

	@Override
	public void on(EventPredicate matcher, Handler handler) {
		throw new UnsupportedOperationException();
	}

	@Override
	public void before(String[] events, String[] entities, int order, Handler handler) {
		for (String event: events) {
			Service destinationService = getDestinationService(event);
			logger.debug("Delegated the composite service '{}' BEFORE handler registration for event '{}' to service '{}'", getName(), event, destinationService.getName());
			destinationService.before(new String[] {event}, entities, order, handler);
		}
	}

	@Override
	public void before(EventPredicate matcher, Handler handler) {
		throw new UnsupportedOperationException();
	}

	@Override
	public void after(String[] events, String[] entities, int order, Handler handler) {
		for (String event: events) {
			Service destinationService = getDestinationService(event);
			logger.debug("Delegated the composite service '{}' AFTER handler registration for event '{}' to service '{}'", getName(), event, destinationService.getName());
			destinationService.after(new String[] {event}, entities, order, handler);
		}
	}

	@Override
	public void after(EventPredicate matcher, Handler handler) {
		throw new UnsupportedOperationException();
	}

	@Override
	public void emit(EventContext context) {
		Service destinationService = getDestinationService(context.getEvent());
		logger.debug("Emitting the event '{}' on service '{}'", context.getEvent(), destinationService.getName());
		destinationService.emit(context);
	}

	private Service getDestinationService(String event) {
		for (Entry<Pattern, Set<String>> entry : eventMatcherToServices.entrySet()) {
			if (entry.getKey().matcher(event).matches()) {
				for (String service : entry.getValue()) {
					Service srv = runtime.getServiceCatalog().getService(service);
					if (srv != null) {
						return srv;
					} else {
						throw new ErrorStatusException(CdsErrorStatuses.NO_SERVICE_IN_CATALOG, service, getName());
					}
				}
			}
		}

		throw new ErrorStatusException(CdsErrorStatuses.NO_DESTINATION_SERVICE, event, getName());
	}

	private void dump() {
		logger.debug("");
		logger.debug("-------- Composite Service '{}' Routing Table --------", getName());
		logger.debug("");
		eventMatcherToServices.forEach((pattern, services) -> {
			logger.debug("    {} -> {}", pattern.pattern(), Arrays.toString(services.toArray()));
		});
		logger.debug("");
	}
}
