/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.draft;

import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;

import com.sap.cds.impl.parser.builder.ExpressionBuilder;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.ElementRef;
import com.sap.cds.ql.Value;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnModifier;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.reflect.CdsBaseType;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.draft.DraftAdministrativeData;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.impl.utils.CdsModelUtils;

/**
 * A {@link CqnModifier} for the all parts of draft statements except for reference, aggregations, orderby and columns.
 */
public class DraftModifier implements CqnModifier {

	private final boolean activeEntityOnly;
	private final boolean forDraftEntity;
	private final CdsEntity entity;
	private final boolean adaptIsActiveEntity;
	private final EventContext context;

	public DraftModifier(CdsEntity entity, boolean adaptIsActiveEntity, EventContext context) {
		this(entity, false, adaptIsActiveEntity, context);
	}

	public DraftModifier(CdsEntity entity, boolean activeEntityOnly, boolean adaptIsActiveEntity, EventContext context) {
		this.entity = entity;
		this.forDraftEntity = entity.getQualifiedName().endsWith(CqnAdapter.DRAFT_SUFFIX);
		this.activeEntityOnly = activeEntityOnly;
		this.adaptIsActiveEntity = adaptIsActiveEntity;
		this.context = context;
	}

	@Override
	public Value<?> ref(ElementRef<?> ref) {
		List<Segment> segments = new ArrayList<>(ref.segments());
		if (referencesDraftField(Drafts.IS_ACTIVE_ENTITY, segments, entity, forDraftEntity) && adaptIsActiveEntity) {
			return CQL.constant(true);
		} else if (!activeEntityOnly) {
			if (DraftAdministrativeData.IN_PROCESS_BY_USER.equals(ref.targetSegment().id())) {
				return createTimeoutExpression(ref);
			}
			if (DraftAdministrativeData.DRAFT_IS_CREATED_BY_ME.equals(ref.targetSegment().id())) {
				return createDraftCreatedByMeExpression(ref);
			}
		}
		return CQL.get(segments);
	}

	private ElementRef<?> changeLastSegment(CqnElementRef ref, String id) {
		List<Segment> result = new ArrayList<>(ref.segments());
		result.set(result.size() - 1, CQL.refSegment(id));
		return CQL.get(result);
	}

	private boolean referencesDraftField(String draftColumn, List<? extends Segment> segments, CdsEntity entity, boolean forDraftEntity) {
		if (segments.stream().anyMatch(s -> s.id().equalsIgnoreCase(draftColumn))) {
			for (Segment segment: segments) {
				String id = segment.id();
				if (draftColumn.equalsIgnoreCase(id)) {
					return !forDraftEntity;
				}
				if (CdsModelUtils.isAssociationToParentOrChild(id, entity)) {
					entity = entity.getTargetOf(id);
				} else if (ReferenceModifier.SIBLING.equals(id) || ReferenceModifier.SIBLING_UNSECURED.equals(id)) {
					forDraftEntity = !forDraftEntity;
				} else if (entity.findAssociation(id).isPresent()) {
					entity = entity.getTargetOf(id);
					forDraftEntity = id.endsWith(CqnAdapter.DRAFT_SUFFIX);
				}
			}
		}

		return false;
	}

	private Value<?> createTimeoutExpression(CqnElementRef ref) {
		Instant timeoutThreshold = getCancellationThreshold(context);
		ElementRef<?> refLastChangeDateTime = changeLastSegment(ref, DraftAdministrativeData.LAST_CHANGE_DATE_TIME);
		ElementRef<?> refDraftUuid = changeLastSegment(ref, DraftAdministrativeData.DRAFT_UUID);
		return (Value<?>) ExpressionBuilder.create()
				.plain("CASE")
				.plain("WHEN")
				.ref(refDraftUuid)
				.plain("IS NULL")
				.plain("THEN")
				.plain("null")
				.plain("WHEN")
				.ref(refLastChangeDateTime)
				.plain(">")
				.add(CQL.val(timeoutThreshold))
				.plain("THEN")
				.ref(ref)
				.plain("ELSE")
				.plain("''")
				.plain("END")
				.value().type(CdsBaseType.STRING.cdsName());
	}

	private Value<?> createDraftCreatedByMeExpression(CqnElementRef ref) {
		String user = context.getUserInfo().getName();
		ElementRef<?> refCreatedByUser = changeLastSegment(ref, DraftAdministrativeData.CREATED_BY_USER);
		return (Value<?>) ExpressionBuilder.create()
				.plain("CASE")
				.plain("WHEN")
				.ref(refCreatedByUser)
				.plain("=")
				.add(CQL.val(user))
				.plain("THEN")
				.add(CQL.constant(true))
				.plain("ELSE")
				.add(CQL.constant(false))
				.plain("END")
				.value().type(CdsBaseType.BOOLEAN.cdsName());
	}

	static Instant getCancellationThreshold(EventContext context) {
		// TODO should we be able to get the NOW time stamp from the context?
		return Instant.now().minus(context.getCdsRuntime().getEnvironment().getCdsProperties().getDrafts().getCancellationTimeout()).truncatedTo(ChronoUnit.MILLIS);
	}
}
