/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.persistence;

import java.time.Instant;
import java.util.HashSet;
import java.util.Map;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import com.sap.cds.impl.DataProcessor;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.model.CdsAnnotations;

/**
 * Event handler for the "managed" aspect.
 */
@ServiceName(value = "*", type = PersistenceService.class)
public class ManagedAspectHandler implements EventHandler {

	/**
	 * Handler method that calculates the administrative data properties of entities
	 * with the "managed" aspect. Usually, these are the properties createdAt / createdBy
	 * and modifiedAt / modifiedBy, but it could be any property annotated with
	 * "@cds.on.insert" / "@cds.on.update" or "@odata.on.insert" / "@odata.on.update" annotations
	 * with matching values.
	 *
	 * @param context the event context
	 */
	@Before(event = { CqnService.EVENT_CREATE, CqnService.EVENT_UPDATE, CqnService.EVENT_UPSERT })
	@HandlerOrder(OrderConstants.Before.CALCULATE_FIELDS)
	public void calculateManagedFields(EventContext context) {
		String event = context.getEvent();

		// TODO should we be able to get the NOW time stamp from the context?
		Instant now = Instant.now();
		String user = context.getUserInfo().getName() == null ? "anonymous" : context.getUserInfo().getName();

		DataProcessor.create().bulkAction((struct, entries) -> {
				// is this a managed entity?
				HashSet<String> userElements = new HashSet<>();
				HashSet<String> nowElements = new HashSet<>();
				findHandledElements(struct.elements(), event, userElements, nowElements);
				if (userElements.isEmpty() && nowElements.isEmpty()) {
					// not managed, nothing to do
					return;
				}

				StreamSupport.stream(entries.spliterator(), true).forEach(map -> {
					// set user element values
					for (String elementName : userElements) {
						if (!map.containsKey(elementName)) {
							map.put(elementName, user);
						}
					}
					// set timestamp element values
					for (String elementName : nowElements) {
						if (!map.containsKey(elementName)) {
							map.put(elementName, now);
						}
					}
				});
			}).process(CdsServiceUtils.getEntities(context), context.getTarget());
	}

	/**
	 * Find elements that must be calculated by this handler in this event.
	 * An element is handled, if
	 * <ul>
	 *   <li>it is annotated with "@cds.on.insert" or "@odata.on.insert" (for CREATE events) or <br>
	 *       it is annotated with "@cds.on.update" or "@odata.on.update" (for UPDATE or UPSERT events) </li>
	 *
	 *   <li>the annotation value is of the kind "=":"$user" or "=":"$now" or <br>
	 *       the annotation value is of the kind "#":"user" or "#":"now </li>
	 * </ul>
	 * @param elements the elements of the target entity
	 * @param event the current event
	 * @param userElements the result set of element names that require setting the value to $user
	 * @param nowElements the result set of element names that require setting the value to $now
	 */
	@SuppressWarnings("unchecked")
	private void findHandledElements(Stream<CdsElement> elements, String event, HashSet<String> userElements, HashSet<String> nowElements) {
		elements.forEach(element -> {

			// TODO: what about creation via UPSERT?
			Object annotationValue = null;
			if (CqnService.EVENT_CREATE.equals(event)) {
				annotationValue = CdsAnnotations.ON_INSERT.getOrDefault(element);
			} else if (CqnService.EVENT_UPDATE.equals(event)) {
				annotationValue = CdsAnnotations.ON_UPDATE.getOrDefault(element);
			} else if (CqnService.EVENT_UPSERT.equals(event)) {
				annotationValue = CdsAnnotations.ON_INSERT.getOrValue(element, CdsAnnotations.ON_UPDATE.getOrDefault(element));
			}

			if (annotationValue instanceof Map) {
				// check for annotations of the kind "@cds.on.insert": {"=": "$user"}
				Object equalsValue = ((Map<String,Object>) annotationValue).get("=");
				if ("$now".equals(equalsValue)) {
					nowElements.add(element.getName());
				}
				if ("$user".equals(equalsValue)) {
					userElements.add(element.getName());
				}
				// check for annotations of the kind "@odata.on.insert": {"#": "user"}
				equalsValue = ((Map<String,Object>) annotationValue).get("#");
				if ("now".equals(equalsValue)) {
					nowElements.add(element.getName());
				}
				if ("user".equals(equalsValue)) {
					userElements.add(element.getName());
				}
			}

		});
	}

}
