/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.persistence;

import java.util.Objects;
import java.util.function.Predicate;

import com.sap.cds.reflect.CdsElement;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.utils.ValidatorUtils;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.model.CdsAnnotations;

/**
 * Handler class to check not null elements. Adding not null to an element leads
 * to a database constraint check. This handler is mainly there to improve the
 * error message in case such an element is set to null
 */
@ServiceName(value = "*", type = PersistenceService.class)
public class NotNullHandler implements EventHandler {

	protected static Predicate<CdsElement> requiresNotNullCheck() {
		return element -> {
			if (!ValidatorUtils.requiresNotNullCheck(element)) {
				return false;
			}
			if (!element.isNotNull() || !CdsAnnotations.ASSERT_NOTNULL.isTrue(element)) {
				// @assert.notNull: false deactivates null check
				return false;
			}
			if (CdsAnnotations.ODATA_FOREIGN_KEY_FOR.getOrValue(element, null) != null) {
				return false;
			}
			return true;
		};
	}

	@Before(event = { CqnService.EVENT_CREATE, CqnService.EVENT_UPDATE, CqnService.EVENT_UPSERT })
	@HandlerOrder(OrderConstants.Before.VALIDATE_FIELDS)
	public void runCheck(EventContext context) {
		ValidatorUtils.runNotNullCheck(context, requiresNotNullCheck(),
				ValidatorUtils.assertNotNull(context, Objects::isNull, (path, element, entity) -> {
					throw new ErrorStatusException(CdsErrorStatuses.VALUE_REQUIRED, element.getName(), entity)
							.messageTarget(path, element);
				}));
	}
}
