/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.request;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import javax.annotation.Nonnull;

import com.sap.cds.services.impl.utils.ModifiedValues;
import com.sap.cds.services.impl.utils.ModifiedValues.MapProperty;
import com.sap.cds.services.impl.utils.ModifiedValues.Property;
import com.sap.cds.services.request.ModifiableUserInfo;
import com.sap.cds.services.request.UserInfo;
import com.sap.cds.services.utils.ClassMethods;

class ModifiableUserInfoImpl implements ModifiableUserInfo {

	private final ModifiedValues modifyValues = new ModifiedValues();

	private final Property<String> id;

	private final Property<String> name;

	private final Property<String> tenant;

	private final Property<Set<String>> roles;

	private final Property<Set<String>> unrestrictedAttributes;

	private final Property<Boolean> systemUser;

	private final Property<Boolean> authenticated;

	private final MapProperty<String, List<String>> attributes;

	private final MapProperty<String, Object> additionalAttributes;

	private final UserInfo prevUserInfo;

	ModifiableUserInfoImpl(@Nonnull UserInfo prevUserInfo) {
		this.prevUserInfo = prevUserInfo;
		this.id = modifyValues.field("id", prevUserInfo::getId);
		this.name = modifyValues.field("name", prevUserInfo::getName);
		this.tenant = modifyValues.field("tenant", prevUserInfo::getTenant);
		this.roles = modifyValues.field("roles", prevUserInfo::getRoles);
		this.unrestrictedAttributes = modifyValues.field("unrestrictedAttributes", prevUserInfo::getUnrestrictedAttributes);
		this.systemUser = modifyValues.field("isSystemUser", prevUserInfo::isSystemUser);
		this.authenticated = modifyValues.field("isAuthenticated", prevUserInfo::isAuthenticated);
		this.attributes = modifyValues.field("attributes", prevUserInfo::getAttributes, prevUserInfo::getAttributeValues);
		this.additionalAttributes = modifyValues.field("additionalAttributes", prevUserInfo::getAdditionalAttributes, prevUserInfo::getAdditionalAttribute);
	}

	@Override
	public String getId() {
		return id.getValue();
	}

	@Override
	public ModifiableUserInfo setId(String val) {
		id.modifyValue(val);
		return this;
	}

	@Override
	public String getName() {
		return name.getValue();
	}

	@Override
	public ModifiableUserInfo setName(String val) {
		name.modifyValue(val);
		return this;
	}

	@Override
	public String getTenant() {
		return tenant.getValue();
	}

	@Override
	public ModifiableUserInfo setTenant(String val) {
		tenant.modifyValue(val);
		return this;
	}

	@Override
	public Set<String> getRoles() {
		return roles.getValue();
	}

	@Override
	public boolean hasRole(String role) {
		if (roles.isModified()) {
			return getRoles().contains(role);
		} else {
			return prevUserInfo.hasRole(role);
		}
	}

	@Override
	public ModifiableUserInfo setRoles(Set<String> val) {
		roles.modifyValue(val);
		return this;
	}

	@Override
	public ModifiableUserInfo addRole(String role) {
		if (roles.getModifiedValue() == null) {
			roles.modifyValue(new TreeSet<>(getRoles())); // make sure we work on modified set
		}
		roles.getModifiedValue().add(role);
		return this;
	}

	@Override
	public ModifiableUserInfo removeRole(String val) {
		if (roles.getModifiedValue() == null) {
			roles.modifyValue(new TreeSet<>(getRoles())); // make sure we work on modified set
		}
		roles.getModifiedValue().remove(val);
		return this;
	}


	@Override
	public List<String> getAttributeValues(String attribute) {
		List<String> result = attributes.getMapValue(attribute);
		return result != null ? result : Collections.emptyList();
	}

	@Override
	public ModifiableUserInfo setAttributeValues(String attribute, List<String> values) {
		attributes.modifyMapValue(attribute, values);
		return this;
	}

	@Override
	public Map<String, List<String>> getAttributes() {
		return attributes.getValue();
	}

	@Override
	public ModifiableUserInfo setAttributes(Map<String, List<String>> val) {
		attributes.modifyValue(val);
		return this;
	}

	@Override
	public Object getAdditionalAttribute(String name) {
		return additionalAttributes.getMapValue(name);
	}

	@Override
	public Map<String, Object> getAdditionalAttributes() {
		return additionalAttributes.getValue();
	}

	@Override
	public ModifiableUserInfo setAdditionalAttributes(Map<String, Object> val) {
		additionalAttributes.modifyValue(val);
		return this;
	}

	@Override
	public ModifiableUserInfo setAdditionalAttribute(String name, Object value) {
		additionalAttributes.modifyMapValue(name, value);
		return this;
	}

	@Override
	public Set<String> getUnrestrictedAttributes() {
		return unrestrictedAttributes.getValue();
	}

	@Override
	public ModifiableUserInfo setUnrestrictedAttributes(Set<String> val) {
		unrestrictedAttributes.modifyValue(val);
		return this;
	}

	@Override
	public ModifiableUserInfo addUnrestrictedAttribute(String val) {
		if (unrestrictedAttributes.getModifiedValue() == null) {
			unrestrictedAttributes.modifyValue(new TreeSet<>(getUnrestrictedAttributes())); // make sure we work on modified set
		}
		unrestrictedAttributes.getModifiedValue().add(val);
		return this;
	}

	@Override
	public ModifiableUserInfo removeUnrestrictedAttribute(String val) {
		if (unrestrictedAttributes.getModifiedValue() == null) {
			unrestrictedAttributes.modifyValue(new TreeSet<>(getUnrestrictedAttributes())); // make sure we work on modified set
		}
		unrestrictedAttributes.getModifiedValue().remove(val);
		return this;
	}

	@Override
	public boolean isAuthenticated() {
		Boolean value = authenticated.getValue();
		return value != null ? value : false;
	}

	@Override
	public ModifiableUserInfo setIsAuthenticated(boolean val) {
		authenticated.modifyValue(val);
		return this;
	}

	@Override
	public boolean isSystemUser() {
		Boolean value = systemUser.getValue();
		return value != null ? value : false;
	}

	@Override
	public ModifiableUserInfo setIsSystemUser(boolean val) {
		systemUser.modifyValue(val);
		return this;
	}

	@Override
	public boolean isPrivileged() {
		// isPrivileged can't be modified
		return prevUserInfo.isPrivileged();
	}

	@Override
	public <T extends UserInfo> T as(Class<T> userInfoClazz) {
		return ClassMethods.as(userInfoClazz, UserInfo.class, this, this::getAdditionalAttributes);
	}
}
