/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.persistence;

import com.sap.cds.SessionContext;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.SessionContextUtils;
import com.sap.cds.util.DataUtils;

import java.util.List;
import java.util.Map;

import static java.util.Objects.isNull;

/**
 * Event handler for the "managed" aspect.
 */
@ServiceName(value = "*", type = PersistenceService.class)
public class ManagedAspectHandler implements EventHandler {

	/**
	 * Handler method that calculates the administrative data properties of entities
	 * with the "managed" aspect. Usually, these are the properties createdAt / createdBy
	 * and modifiedAt / modifiedBy, but it could be any property annotated with
	 * "@cds.on.insert" / "@cds.on.update" or "@odata.on.insert" / "@odata.on.update" annotations
	 * with matching values.
	 *
	 * @param context the event context
	 */
	@Before(event = { CqnService.EVENT_CREATE, CqnService.EVENT_UPDATE, CqnService.EVENT_UPSERT })
	@HandlerOrder(OrderConstants.Before.CALCULATE_FIELDS)
	public void calculateManagedFields(EventContext context) {
		SessionContext sessionContext = SessionContextUtils.toSessionContext(context);
		if (isNull(sessionContext.getUserContext().getId())) {
			sessionContext.getUserContext().setId("anonymous");
		}
		DataUtils dataUtils = DataUtils.create(() -> sessionContext, 7);

		String event = context.getEvent();
		List<Map<String, Object>> entries = CdsServiceUtils.getEntities(context);
		CdsStructuredType struct = context.getTarget();

		if (CqnService.EVENT_CREATE.equals(event)) {
			dataUtils.processOnInsert(struct, entries);
		} else if (CqnService.EVENT_UPDATE.equals(event)) {
			dataUtils.processOnUpdate(struct, entries, true);
		} else if (CqnService.EVENT_UPSERT.equals(event)) {
			// Upsert populates the @cds.on.insert and @cds.on.update annotated fields
			dataUtils.processOnInsert(struct, entries);
			dataUtils.processOnUpdate(struct, entries, true);
		}
	}
}
