package com.sap.cds.services.impl.outbox.persistence;

import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.lang3.RandomUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.ql.Insert;
import com.sap.cds.services.changeset.ChangeSetContext;
import com.sap.cds.services.changeset.ChangeSetListener;
import com.sap.cds.services.impl.auditlog.AuditLogDefaultOutboxOnHandler;
import com.sap.cds.services.impl.outbox.Messages;
import com.sap.cds.services.impl.outbox.Messages_;
import com.sap.cds.services.impl.outbox.persistence.collectors.PartitionCollectorCoordinator;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.outbox.OutboxMessageEventContext;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.messaging.MessagingOutboxUtils;
import com.sap.cds.services.utils.outbox.AbstractOutboxService;

public class PersistentOutbox extends AbstractOutboxService {

	private static final Logger LOG = LoggerFactory.getLogger(PersistentOutbox.class);

	private final CdsRuntime runtime;
	private PartitionCollectorCoordinator collector;

	// in order to keep single registration for the same change set
	private Map<ChangeSetContext, Set<Integer>> changeSetCache = new ConcurrentHashMap<>();

	public PersistentOutbox(String name, CdsRuntime runtime) {
		super(name);
		this.runtime = runtime;
	}

	void init() {
		if (collector == null) {
			collector = new PartitionCollectorCoordinator(runtime);
		}
	}

	void stop() {
		if (collector != null) {
			collector.stop();
		}
	}

	@Override
	protected void enroll(OutboxMessageEventContext context) {
		LOG.debug("Enrolling outbox message with target event '{}'.", context.getEvent());
		int partition = persist(context);
		LOG.debug("Stored outbox message with target event '{}' in partition '{}'.", context.getEvent(), partition);

		if (context.getCdsRuntime().getEnvironment().getCdsProperties().getOutbox().getPersistent().getTriggerSchedule()
				.isEnabled() && !changeSetCache.getOrDefault(context.getChangeSetContext(), Collections.emptySet()).contains(partition)) {

			ChangeSetContext changeSetContext = context.getChangeSetContext();
			changeSetCache.compute(context.getChangeSetContext(), (k, v) ->  {
				if(v == null) {
					v = new HashSet<>();
				}
				v.add(partition);
				return v;
			});

			// schedule the outbox at the end of transaction
			context.getChangeSetContext().register(new ChangeSetListener() {
				@Override
				public void afterClose(boolean completed) {
					Set<Integer> partitions = changeSetCache.remove(changeSetContext);
					if (completed && collector != null) {
						partitions.forEach(collector::schedule);
					}
				}
			});
		}
	}

	private int persist(OutboxMessageEventContext context) {
		Messages message = Messages.create();
		message.setMsg(context.getMessage());
		String target = context.getEvent();
		message.setTarget(target);
		message.setTimestamp(context.getTimestamp());

		int partition;
		if (target.equals(AuditLogDefaultOutboxOnHandler.OUTBOX_AUDITLOG_TARGET)) {
			partition = 1;
		} else if (target.startsWith(MessagingOutboxUtils.OUTBOX_MESSAGING_TARGET)) {
			partition = 0;
		} else {
			partition = RandomUtils.nextInt(0, PartitionCollectorCoordinator.OUTBOX_MESSAGE_PARTITIONS);
		}

		message.setPartition(partition);

		PersistenceService db = CdsServiceUtils.getDefaultPersistenceService(context);
		Insert insert = Insert.into(Messages_.class).entry(message);
		db.run(insert);

		return partition;
	}

}
