/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.runtime;

import static com.sap.cds.services.impl.request.ParameterInfoFactory.emptyParameterInfo;
import static com.sap.cds.services.impl.request.UserInfoFactory.anonymousUserInfo;
import static com.sap.cds.services.impl.request.UserInfoFactory.privilegedUserInfo;

import java.util.Map;
import java.util.Objects;
import java.util.function.Consumer;
import java.util.function.Function;

import org.apache.commons.lang3.StringUtils;

import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.ServiceCatalog;
import com.sap.cds.services.authentication.AuthenticationInfo;
import com.sap.cds.services.impl.messages.MessagesImpl;
import com.sap.cds.services.impl.request.RequestContextImpl;
import com.sap.cds.services.impl.request.RequestContextSPI;
import com.sap.cds.services.messages.Messages;
import com.sap.cds.services.request.FeatureTogglesInfo;
import com.sap.cds.services.request.ModifiableParameterInfo;
import com.sap.cds.services.request.ModifiableUserInfo;
import com.sap.cds.services.request.ParameterInfo;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.request.UserInfo;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.runtime.RequestContextRunner;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.CorrelationIdUtils;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.TenantIdUtils;

public class RequestContextRunnerImpl implements RequestContextRunner {

	/**
	 * A reference to the active runtime.
	 */
	private final CdsRuntime cdsRuntime;

	/**
	 * parentRequestContext is null in case the runner creates the initial RequestContext of a request.
	 */
	private final RequestContextSPI parentRequestContext;
	private final String parentTenant;

	private UserInfo userInfo;
	private ParameterInfo parameterInfo;
	private FeatureTogglesInfo featureTogglesInfo;
	private AuthenticationInfo authenticationInfo;
	private boolean clearMessages = false;

	public RequestContextRunnerImpl(CdsRuntime cdsRuntime) {
		this.cdsRuntime = Objects.requireNonNull(cdsRuntime, "cdsRuntime must not be null");
		this.parentRequestContext = RequestContextImpl.getCurrentInternal();

		// ensures that provider-dependent info objects are always initialized as part of the parent thread
		// this comes at the cost of being potentially unnecessary, in case UserInfo or ParameterInfo objects are directly passed to the runner
		// however the benefit of usability of the API outweighs these drawbacks
		if (parentRequestContext != null) {
			parentTenant = parentRequestContext.getUserInfo().getTenant();
			userInfo = parentRequestContext.getUserInfo();
			parameterInfo = parentRequestContext.getParameterInfo();
			featureTogglesInfo = parentRequestContext.getFeatureTogglesInfo();
			authenticationInfo = parentRequestContext.getAuthenticationInfo();
		} else {
			parentTenant = null;
			userInfo = providedUserInfo();
			parameterInfo = providedParameterInfo();
			featureTogglesInfo = null; // handle toggles lazily on basis of fixed parameters or explicit input
			authenticationInfo = cdsRuntime.getProvidedAuthenticationInfo();
		}
		assert userInfo != null;
		assert parameterInfo != null;
	}

	@Override
	public RequestContextRunner user(UserInfo userInfo) {
		Objects.requireNonNull(userInfo, "userInfo must not be null");
		this.userInfo = userInfo;
		return this;
	}

	@Override
	public RequestContextRunner anonymousUser() {
		userInfo = anonymousUserInfo();
		return this;
	}

	@Override
	public RequestContextRunner providedUser() {
		userInfo = providedUserInfo();
		return this;
	}

	@Override
	public RequestContextRunner privilegedUser() {
		String tenant = userInfo.getTenant();
		Map<String, Object> additionalAttributes = userInfo.getAdditionalAttributes();
		userInfo = privilegedUserInfo();
		// automatically propagate tenant & additional attributes for privileged users
		modifyUser(u -> u.setTenant(tenant).setAdditionalAttributes(additionalAttributes));
		return this;
	}

	@Override
	public RequestContextRunner modifyUser(Consumer<ModifiableUserInfo> contextUser) {
		UserInfo prevUserInfo = userInfo;

		ModifiableUserInfo modifiedUser = prevUserInfo.copy();
		if (contextUser != null) {
			contextUser.accept(modifiedUser);
		}
		userInfo = modifiedUser;
		return this;
	}

	@Override
	public RequestContextRunner parameters(ParameterInfo parameterInfo) {
		Objects.requireNonNull(parameterInfo, "parameterInfo must not be null");
		this.parameterInfo = parameterInfo;
		return this;
	}

	@Override
	public RequestContextRunner featureToggles(FeatureTogglesInfo featureTogglesInfo) {
		Objects.requireNonNull(featureTogglesInfo, "featureTogglesInfo must not be null");
		// feature toggles may be only set explicitly for initial RequestContext
		if (this.parentRequestContext == null) {
			this.featureTogglesInfo = featureTogglesInfo;
		} else {
			throw new ErrorStatusException(CdsErrorStatuses.FEATURETOGGLESINFO_OVERRIDE);
		}

		return this;
	}

	@Override
	public RequestContextRunner clearParameters() {
		parameterInfo = emptyParameterInfo();
		return this;
	}

	@Override
	public RequestContextRunner providedParameters() {
		parameterInfo = providedParameterInfo();
		return this;
	}

	@Override
	public RequestContextRunner modifyParameters(Consumer<ModifiableParameterInfo> contextParamters) {
		ParameterInfo prevParameterInfo = parameterInfo;

		ModifiableParameterInfo modifiedParameter = prevParameterInfo.copy();
		if (contextParamters != null) {
			contextParamters.accept(modifiedParameter);
		}
		parameterInfo = modifiedParameter;

		return this;
	}

	@Override
	public RequestContextRunner clearMessages() {
		clearMessages = true;
		return this;
	}

	@Override
	public void run(Consumer<RequestContext> requestHandler) {
		run((requestContext) -> {
			requestHandler.accept(requestContext);
			return Void.TYPE;
		});
	}

	@Override
	public <T> T run(Function<RequestContext, T> requestHandler) {
		ServiceCatalog serviceCatalog = cdsRuntime.getServiceCatalog(); // could be exposed by a provider later

		// calculate featureTogglesInfo if not set explicitly already.
		if (featureTogglesInfo == null) {
			// userInfo and parameterInfo are fixed and thread-safe.
			featureTogglesInfo = calculateFeatureTogglesInfo();
		}

		// determine model only initially or if tenant changed
		CdsModel cdsModel;
		if(parentRequestContext == null || !Objects.equals(parentTenant, userInfo.getTenant())) {
			cdsModel = cdsRuntime.getCdsModel(userInfo, featureTogglesInfo);
		} else {
			cdsModel = parentRequestContext.getModel();
		}

		Messages messages;
		if (!clearMessages && parentRequestContext != null) {
			messages = parentRequestContext.getMessages();
		} else {
			messages = new MessagesImpl(cdsRuntime, parameterInfo.getLocale());
		}
		boolean putTenantInMDC = !TenantIdUtils.mdcHasEntry();
		boolean putInMDC = !CorrelationIdUtils.mdcHasEntry();

		String prevTenant = null;
		try (RequestContextSPI requestContext = RequestContextImpl.open(cdsModel, serviceCatalog, userInfo,
				parameterInfo, featureTogglesInfo, messages, authenticationInfo)) {

			// correlation id is read from the thread that created the RequestContextRunner
			// instance that is passed to the new thread to invoke run
			String correlationId = parameterInfo.getCorrelationId();
			if (putInMDC) {
				CorrelationIdUtils.putInMDC(correlationId);
			} else if (!StringUtils.isEmpty(correlationId) && !CorrelationIdUtils.getFromMDC().equals(correlationId)) {
				throw new ErrorStatusException(CdsErrorStatuses.MULTIPLE_CORRELATION_IDS, CorrelationIdUtils.getFromMDC(), correlationId);
			}
			String tenantId = userInfo.getTenant();
			if (putTenantInMDC) {
				TenantIdUtils.putInMDC(tenantId);
			} else if (!TenantIdUtils.getFromMDC().equals(tenantId)) {
				prevTenant = TenantIdUtils.getFromMDC();
				TenantIdUtils.putInMDC(tenantId);
			}
			return requestHandler.apply(requestContext);
		} finally {
			if (putInMDC) {
				CorrelationIdUtils.clearMDC();
			}
			if (putTenantInMDC) {
				TenantIdUtils.clearMDC();
			} else if (prevTenant != null) {
				TenantIdUtils.putInMDC(prevTenant);
			}
		}
	}

	private UserInfo providedUserInfo() {
		UserInfo result = cdsRuntime.getProvidedUserInfo();
		if (result != null) {
			return result;
		}
		return anonymousUserInfo();
	}

	private ParameterInfo providedParameterInfo() {
		ParameterInfo result = cdsRuntime.getProvidedParameterInfo();
		if (result != null) {
			return result;
		}
		return emptyParameterInfo();
	}

	private FeatureTogglesInfo calculateFeatureTogglesInfo() {
		FeatureTogglesInfo result = cdsRuntime.getFeatureTogglesInfo(userInfo, parameterInfo);
		if (result != null) {
			return result;
		}
		return FeatureTogglesInfo.create();
	}

}
