/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.utils;

import java.util.Collection;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

public class CombinedMap<K, V> implements Map<K, V> {

	private final Map<K, V> modified;
	private final Map<K, V> basis;

	public CombinedMap(Map<K, V> modified, Map<K, V> basis) {
		this.modified = modified;
		this.basis = basis;
	}

	@Override
	public int size() {
		return keySet().size();
	}

	@Override
	public boolean isEmpty() {
		return modified.isEmpty() && basis.isEmpty();
	}

	@Override
	public boolean containsKey(Object key) {
		return modified.containsKey(key) || basis.containsKey(key);
	}

	@Override
	public boolean containsValue(Object value) {
		return modified.containsValue(value) || basis.containsValue(value);
	}

	@Override
	public V get(Object key) {
		if (modified.containsKey(key)) {
			return modified.get(key);
		} else {
			return basis.get(key);
		}
	}

	@Override
	public V put(K key, V value) {
		return modified.put(key, value);
	}

	@SuppressWarnings("unchecked")
	@Override
	public V remove(Object key) {
		return modified.put((K)key, null); // basis should be invisible
	}

	@Override
	public void putAll(Map<? extends K, ? extends V> m) {
		modified.putAll(m);
	}

	@Override
	public void clear() {
		for (K key : keySet()) {
			remove(key);
		}
	}

	@Override
	public Set<K> keySet() {
		Set<K> combined = new HashSet<>(modified.keySet());
		combined.addAll(basis.keySet());
		return combined;
	}

	@Override
	public Collection<V> values() {
		return entrySet().stream().map(e -> e.getValue()).collect(Collectors.toList());
	}

	@Override
	public Set<Entry<K, V>> entrySet() {
		Set<Entry<K, V>> combined = new HashSet<>(modified.entrySet());
		basis.entrySet().stream().filter(e -> !modified.keySet().contains(e.getKey())).forEach(combined::add);
		return combined;
	}

}
