/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.draft;

import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ThreadLocalRandom;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.services.application.ApplicationLifecycleService;
import com.sap.cds.services.application.ApplicationPreparedEventContext;
import com.sap.cds.services.application.ApplicationStoppedEventContext;
import com.sap.cds.services.environment.CdsProperties.Drafts.GC;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.DraftUtils;

/**
 * @author d064456
 *
 */
@ServiceName(value = ApplicationLifecycleService.DEFAULT_NAME)
public class DraftGCTimerInitializer implements EventHandler {

	private static final Logger log = LoggerFactory.getLogger(DraftGCTimerInitializer.class);

	private final CdsRuntime runtime;
	private Timer timer;

	public DraftGCTimerInitializer(CdsRuntime runtime) {
		this.runtime = runtime;
	}

	@On
	protected void initializeGC(ApplicationPreparedEventContext context) {
		GC gcConfig = runtime.getEnvironment().getCdsProperties().getDrafts().getGc();
		if (gcConfig.isEnabled() && timer == null) {
			timer = new Timer("Draft GC Timer", true);
			long gcInterval = gcConfig.getInterval().toMillis();
			// distribute the first execution time randomly to reduce the database load when e.g. restarting the app
			long firstExecutionTime = ThreadLocalRandom.current().nextLong(gcInterval);
			timer.schedule(new TimerTask() {
				@Override
				public void run() {
					try {
						DraftUtils.gcDraftsOfAllServicesAndTenants(runtime);
					} catch (Throwable t) {
						log.error("Failed to gc drafts", t);
					}
				}
			}, firstExecutionTime, gcInterval);
		}
	}

	@On
	protected void stopGC(ApplicationStoppedEventContext context) {
		if(timer != null) {
			timer.cancel();
			timer = null;
		}
	}

}
