/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.handlerregistry;

import java.text.MessageFormat;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;

import com.sap.cds.services.handler.EventPredicate;
import com.sap.cds.services.utils.StringUtils;

public class EventPredicateTools {

	private EventPredicateTools() {
		// no instances
	}

	public static EventPredicate create(String[] eventNames, String[] entityNames) {
		Collection<String> evCol = toMatchCollection(eventNames);
		Collection<String> etCol = toMatchCollection(entityNames);

		if (evCol.size() > 1 || etCol.size() > 1) {
			return new EventPredicateCollection(evCol, etCol);
		} else {
			return create(firstOf(evCol), firstOf(etCol));
		}
	}

	public static EventPredicate create(String event, String entity) {
		event = toMatchString(event);
		entity = toMatchString(entity);
		return event == null && entity == null ? EventPredicate.ALL : new EventPredicateString(event, entity);
	}

	private static final class EventPredicateString implements EventPredicate {
		private final String event;
		private final String entity;

		public EventPredicateString(String event, String entity) {
			this.event = event;
			this.entity = entity;
		}

		@Override
		public boolean test(String event, String entity) {
			// empty collection / null is treated as wildcard
			return (this.event == null || this.event.equals(event))
					&& (this.entity == null || this.entity.equals(entity));
		}

		@Override
		public String toString() {
			return MessageFormat.format("event ''{0}'' on entity ''{1}''",
					StringUtils.notEmpty(event, "*"), StringUtils.notEmpty(entity, "*"));
		}
	}

	private static final class EventPredicateCollection implements EventPredicate {
		private final Collection<String> events;
		private final Collection<String> entities;

		public EventPredicateCollection(Collection<String> events, Collection<String> entities) {
			this.events = events.isEmpty() ? null : events;
			this.entities = entities.isEmpty() ? null : entities;
		}

		@Override
		public boolean test(String event, String entity) {
			// empty collection / null is treated as wildcard
			return (this.events == null || this.events.contains(event)) && //
					(this.entities == null || this.entities.contains(entity));
		}

		@Override
		public String toString() {
			return MessageFormat.format("events ''{0}'' on entities ''{1}''",
					(events != null && !events.isEmpty() ? events : "*"),
					(entities != null && !entities.isEmpty() ? entities : "*"));
		}
	}

	/**
	 * Returns a collection of the strings present in the data array. If one of the
	 * strings is either "*" or "" an empty collection is returned. Null values are
	 * stripped from the collection.
	 *
	 * An empty collection is meant to be treated as a wildcard. A collection with
	 * at least one element should be treated as a list of possible options.
	 *
	 * @param data the original data
	 * @return the cleansed collection
	 */
	static Collection<String> toMatchCollection(String[] data) {
		if (data == null || data.length == 0) {
			return Collections.emptySet();
		}

		if (data.length == 1) {
			String str = toMatchString(data[0]);
			return str == null ? Collections.emptySet() : Collections.singleton(str);
		}

		HashSet<String> res = new HashSet<String>();
		Collections.addAll(res, data);
		if (res.contains("*") || res.contains("")) {
			return Collections.emptySet();
		}
		res.remove(null);

		switch (res.size()) {
		case 0:
			return Collections.emptySet();
		case 1:
			return Collections.singleton(res.iterator().next());
		default:
			return res;
		}
	}

	private static <T> T firstOf(Collection<T> coll) {
		return coll.isEmpty() ? null : coll.iterator().next();
	}

	/**
	 * Cleanses the string, treating null, "" and "*" as wildcard
	 *
	 * @param data the original data
	 * @return the cleansed string
	 */
	private static String toMatchString(String data) {
		if (data == null || data.isEmpty() || "*".equals(data)) {
			return null;
		}
		return data;
	}

}
