/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.model;

import static java.util.stream.Collectors.joining;

import java.util.Set;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.mtx.MetaDataAccessor;
import com.sap.cds.mtx.ModelId;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.request.FeatureTogglesInfo;
import com.sap.cds.services.request.UserInfo;
import com.sap.cds.services.runtime.CdsModelProvider;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.model.DynamicModelUtils;

/**
 * A {@link CdsModelProvider} that uses the Sidecar to retrieve tenant dependent models.
 */
public class DynamicModelProvider implements CdsModelProvider {

	private static final Logger logger = LoggerFactory.getLogger(DynamicModelProvider.class);

	private final CdsRuntime runtime;
	private final MetaDataAccessor<String> accessor;
	private final boolean modelProviderEnabled;

	public DynamicModelProvider(CdsRuntime runtime) {
		this.runtime = runtime;
		DynamicModelUtils utils = new DynamicModelUtils(runtime);
		this.accessor = utils.createMetadataAccessor((edmx, service) -> edmx);
		this.modelProviderEnabled = utils.isModelProviderEnabled();
	}

	@Override
	public CdsModel get(UserInfo userInfo, FeatureTogglesInfo featureTogglesInfo) {
		String tenant = userInfo.getTenant();
		boolean hasAllToggles = featureTogglesInfo.getEnabledFeatureToggles().anyMatch(t -> t.getName().trim().equals("*"));
		// as of V3 tenant can be null -> e.g. single-tenant feature toggles scenario
		if(tenant == null && (!modelProviderEnabled || hasAllToggles)) {
			return runtime.getCdsModel();
		} else {
			logger.debug("Fetching the model for tenant '{}' and feature toggles '{}'", tenant,
					featureTogglesInfo.getFeatureToggles().map(ft -> ft.getName() + ": " + ft.isEnabled())
					.collect(joining(",")));
			Set<String> features = featureTogglesInfo.getEnabledFeatureToggles().map(ft -> ft.getName()).collect(Collectors.toSet());
			ModelId modelId = ModelId.create(tenant).features(features).build();
			int maxAgeSeconds = Integer.MAX_VALUE;
			return accessor.getCdsModel(modelId, maxAgeSeconds);
		}
	}
}
