/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.persistence;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UncheckedIOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.base.Charsets;
import com.sap.cds.impl.parser.StructDataParser;
import com.sap.cds.ql.Insert;
import com.sap.cds.reflect.CdsArrayedType;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsSimpleType;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.services.ServiceException;
import com.sap.cds.services.changeset.ChangeSetContext;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.util.CdsModelUtils;
import com.sap.cds.util.CdsTypeUtils;
import com.sap.cds.util.DataUtils;

import org.apache.commons.io.input.BOMInputStream;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static com.sap.cds.reflect.CdsBaseType.UUID;
public class CsvDataLoader {

	private final static Logger logger = LoggerFactory.getLogger(CsvDataLoader.class);

	private final PersistenceService db;
	private final CdsRuntime runtime;

	public CsvDataLoader(PersistenceService db, CdsRuntime runtime) {
		this.db = db;
		this.runtime = runtime;
	}

	/**
	 * Loads CSVs from the first existing CSV folder location.
	 * The paths is resolved based on the current working directory
	 */
	public void load() {
		if (runtime.getEnvironment().getCdsProperties().getDataSource().isCsvSingleChangeset()) {
			// Run the whole CSV data import in a single changeset to avoid possible foreign key constraint violations,
			// if assert_integrity in DB is enabled
			runtime.changeSetContext().run(t -> {
				load(t);
			});
		} else {
			// import each CSV file in it's own changeset
			load(null);
		}
	}

	private void load(ChangeSetContext ctx) {
		for(String path : runtime.getEnvironment().getCdsProperties().getDataSource().getCsvPaths()) {
			boolean recursive = false;
			if(path.endsWith("**") && path.length() > 2) {
				path = path.substring(0, path.length() - 2);
				recursive = true;
			}

			File folder = new File(path);
			if(folder.exists() && folder.isDirectory()) {
				loadFolder(folder, recursive, ctx);
			}
		}
	}

	/**
	 * Loads CSV files from within the specified folder
	 * @param folder the folder
	 * @param recursive true, if the folder should be scanned recursively
	 * @param ctx an optional {@link ChangeSetContext}
	 */
	public void loadFolder(File folder, boolean recursive, ChangeSetContext ctx) {
		String fileSuffix = runtime.getEnvironment().getCdsProperties().getDataSource().getCsvFileSuffix();
		for (File file : folder.listFiles()) {
			if (file.isFile() && file.getName().endsWith(fileSuffix)) {
				loadFile(file, ctx);
			}

			if (recursive && file.isDirectory()) {
				loadFolder(file, true, ctx);
			}
		}
	}

	/**
	 * Loads CSV files from the specified file (extension needs to be .csv)
	 * @param file the file
	 * @param ctx an optional {@link ChangeSetContext}
	 */
	void loadFile(File file, ChangeSetContext ctx) {
		if (file.length() == 0) {
			return;
		}
		try (CSV csv = new CSV(file)) {
			List<Map<String, Object>> entries = csv.data().collect(Collectors.toList());
			if (!entries.isEmpty()) {
				if (ctx == null) {
					try {
						runtime.changeSetContext().run(context -> {
							loadFile(csv, entries, context);
						});
					} catch (ServiceException e) {
						logger.debug("Skipped filling {} from {}", csv.entity.getQualifiedName(), file.getPath(), e);
					}
				} else {
					loadFile(csv, entries, ctx);
				}
			}
		} catch (UncheckedIOException | IOException e) {
			throw new ErrorStatusException(CdsErrorStatuses.INVALID_CSV_FILE, file.getPath(), e); // NOSONAR
		}
	}

	private void loadFile(CSV csv, List<Map<String, Object>> entries, ChangeSetContext ctx) {
		Insert insert = Insert.into(csv.entity()).entries(entries);
		db.run(insert);
		if(ctx.isMarkedForCancel()) {
			logger.debug("Cancelled filling {} from {}", csv.entity().getQualifiedName(), csv.file.getPath());
		} else {
			logger.info("Filling {} from {}", csv.entity().getQualifiedName(), csv.file.getPath());
		}
	}

	private class CSV implements AutoCloseable {

		private final CdsEntity entity;
		private final File file;
		private final BufferedReader br;
		private final List<ElementPath> headers;

		private String separator = ";";

		public CSV(File file) throws IOException {
			this.file = file;
			br = new BufferedReader(
					new InputStreamReader(new BOMInputStream(new FileInputStream(file)), Charsets.UTF_8));

			String entityName = entityName(file.getName());
			entity = runtime.getCdsModel().findEntity(entityName).orElseGet(() -> {
				String theEntityName = entityName;
				if (theEntityName.lastIndexOf(".texts") > 0) {
					Optional<CdsEntity> entity = runtime.getCdsModel().findEntity(
							theEntityName.substring(0, theEntityName.lastIndexOf(".texts") + 6));
					if (entity.isPresent()) {
						return entity.get();
					}
				}
				if (theEntityName.lastIndexOf("_texts") > 0) {
					theEntityName = theEntityName.substring(0, theEntityName.lastIndexOf("_texts") + 6);
					Optional<CdsEntity> entity = runtime.getCdsModel().findEntity(theEntityName);
					if (entity.isPresent()) {
						return entity.get();
					}
				}
				if (theEntityName.endsWith("_texts")) {
					// replace entity suffix "_texts" with ".texts"
					String altEntityName = theEntityName.substring(0, theEntityName.length() - "_texts".length()) + ".texts";
					return runtime.getCdsModel().findEntity(altEntityName).orElse(null);
				}
				return null;
			});
			if (entity == null) {
				throw new ErrorStatusException(CdsErrorStatuses.INVALID_CSV_FILE_ENTITYNOTFOUND, entityName, file.getPath()); // NOSONAR
			}

			String headerLine;
			while ((headerLine = br.readLine()) != null && headerLine.isEmpty()) {}
			if (headerLine == null) {
				// file is empty -> just ignore it
				headers = Collections.emptyList();
				return;
			}
			if ( headerLine.indexOf(',') >= 0 ) { // we presume header titles don't have chars like "," or ";"
				if (headerLine.indexOf(';') >= 0) {
					throw new ErrorStatusException(CdsErrorStatuses.INVALID_CSV_FILE_INVALIDHEADER, file.getPath()); // NOSONAR
				}
				separator = ",";
			}

			headers = Arrays.stream(headerLine.split(separator)).map(n -> path(entity, n.trim())).collect(Collectors.toList());
		}

		public CdsEntity entity() {
			return entity;
		}

		public Stream<Map<String, Object>> data() {
			if(headers.isEmpty()) {
				return Stream.empty();
			}
			return br.lines().map(this::data);
		}

		private Map<String, Object> data(String dataLine) {
			Map<String, Object> row = new HashMap<>();
			String[] values = dataLine.split(separator);
			for (int i = 0; i < headers.size(); i++) {
				ElementPath ep = headers.get(i);
				CdsElement element = ep.element;
				String path = ep.path;
				String value = i >= values.length ? "": values[i];

				if (StringUtils.isNotEmpty(value)) {
					handleValue(row, element, path, value);
				}
			}
			return row;
		}

		private void handleValue(Map<String, Object> row, CdsElement element, String path, String value) {
			if (value.startsWith("\"") && value.endsWith("\"")) {
				value = value.substring(1, value.length() - 1);
			}
			CdsType type = element.getType();
			if (type.isSimpleType(UUID) && !CdsTypeUtils.isStrictUUID(element, type)) {
				direct(row, path, type.as(CdsSimpleType.class), value);
			} else if (type.isSimple()) {
				simple(row, path, type.as(CdsSimpleType.class), value);
			} else if (type.isStructured()) {
				structured(row, path, type.as(CdsStructuredType.class), value);
			} else if (type.isArrayed()) {
				arrayed(row, path, element, value);
			} else if (type.isAssociation()) {
				association(row, path, element, value);
			}
		}

		private void direct(Map<String, Object> row, String path, CdsSimpleType type, String value) {
			try {
				DataUtils.resolvePathAndAdd(row, path, value);
			} catch (Exception e) {
				throw error(path, type, false, value, e);
			}
		}

		private void simple(Map<String, Object> row, String path, CdsSimpleType type, String value) {
			try {
				Object val = CdsTypeUtils.parse(type.getType(), value); 
				DataUtils.resolvePathAndAdd(row, path, val);
			} catch (Exception e) {
				throw error(path, type, false, value, e);
			}
		}

		private void structured(Map<String, Object> row, String path, CdsStructuredType type, String value) {
			try {
				Map<String, Object> val = StructDataParser.create(type).parseObject(value);
				DataUtils.resolvePathAndAdd(row, path, val);
			} catch (Exception e) {
				throw error(path, type, false, value, e);
			}
		}

		private void arrayed(Map<String, Object> row, String path, CdsElement element, String value) {
			CdsType itemsType = element.getType().as(CdsArrayedType.class).getItemsType();
			try {
				Object val = StructDataParser.parseArrayOf(itemsType, value);
				DataUtils.resolvePathAndAdd(row, path, val);
			} catch (Exception e) {
				throw error(path, itemsType, true, value, e);
			}
		}

		private void association(Map<String, Object> row, String path, CdsElement element, String value) {
			CdsEntity targetType = element.getType().as(CdsAssociationType.class).getTarget();
			StructDataParser parser = StructDataParser.create(targetType);
			boolean isSingleValued = CdsModelUtils.isSingleValued(element.getType());
			try {
				Object val = isSingleValued ? parser.parseObject(value) : parser.parseArray(value);
				DataUtils.resolvePathAndAdd(row, path, val);
			} catch (Exception e) {
				throw error(path, targetType, !isSingleValued, value, e);
			}
		}

		private ErrorStatusException error(String element, CdsType type, boolean array, String value, Exception e) {
			return new ErrorStatusException(CdsErrorStatuses.INVALID_CSV_FILE_TYPEMISMATCH, value,
					array ? type.getName() + "[]" : type.getName(), element, file.getPath(), e); // NOSONAR
		}

		private String entityName(String fileName) {
			return fileName.replace("-", ".").substring(0, fileName.length() - 4);
		}

		private ElementPath path(CdsStructuredType entity, String column) {
			Optional<ElementPath> path = findElement(entity, column).map(p -> new ElementPath(entity.getElement(p), p));
			return path.orElseThrow(() -> new ErrorStatusException(CdsErrorStatuses.INVALID_CSV_FILE_UNKNOWNCOLUMN,
					file.getPath(), column, entity.getQualifiedName()));
		}

		private Optional<String> findElement(CdsStructuredType entity, String column) {
			column = column.toUpperCase(Locale.US);
			Map<String, CdsElement> normalized = normalize(entity);
			
			CdsElement element = normalized.get(column);
			if (element != null) {
				return Optional.of(element.getName());
			}

			for (Map.Entry<String, CdsElement> entry : normalized.entrySet()) {
				String n = entry.getKey();
				CdsElement e = entry.getValue();
				if (column.startsWith(n + "_")) {
					String suffix = column.substring(n.length() + 1);
					CdsType type = e.getType();
					Optional<String> subElement = Optional.empty();
					if (type.isStructured()) {
						subElement = findElement(type.as(CdsStructuredType.class), suffix);
					} else if (CdsModelUtils.managedToOne(type)) {
						subElement = findElement(type.as(CdsAssociationType.class).getTarget(), suffix);
					}
					if (subElement.isPresent()) {
						String path = e.getName() + "." + subElement.get();
						return Optional.of(path);
					}
				}
			}

			return Optional.empty();
		}

		private Map<String, CdsElement> normalize(CdsStructuredType type) {
			return type.elements().collect(Collectors.toMap(e -> e.getName().toUpperCase(Locale.US), e -> e));
		}

		@Override
		public void close() throws IOException {
			if (br != null)
				br.close();
		}
		
	}

	private static class ElementPath {
		final CdsElement element;
		final String path;
		
		public ElementPath(CdsElement element, String path) {
			this.element = element;
			this.path = path;
		}
	}
}
