/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.draft;

import static com.sap.cds.services.draft.DraftAdministrativeData.DRAFT_UUID;
import static com.sap.cds.services.draft.DraftAdministrativeData.LAST_CHANGE_DATE_TIME;
import static com.sap.cds.services.draft.DraftService.EVENT_DRAFT_CANCEL;
import static com.sap.cds.services.draft.DraftService.EVENT_DRAFT_EDIT;
import static com.sap.cds.services.draft.DraftService.EVENT_DRAFT_NEW;
import static com.sap.cds.services.draft.DraftService.EVENT_DRAFT_PATCH;
import static com.sap.cds.services.draft.DraftService.EVENT_DRAFT_PREPARE;
import static com.sap.cds.services.draft.DraftService.EVENT_DRAFT_SAVE;
import static com.sap.cds.services.draft.Drafts.DRAFT_ADMINISTRATIVE_DATA_DRAFT_UUID;
import static com.sap.cds.services.utils.model.CqnUtils.andPredicate;
import static com.sap.cds.services.utils.model.CqnUtils.modifiedWhere;
import static com.sap.cds.util.CdsModelUtils.entity;
import static com.sap.cds.util.CqnStatementUtils.resolveStar;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;

import java.text.Collator;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.sap.cds.CdsDataProcessor;
import com.sap.cds.CdsDataProcessor.Converter;
import com.sap.cds.Result;
import com.sap.cds.ResultBuilder;
import com.sap.cds.Row;
import com.sap.cds.Struct;
import com.sap.cds.impl.DataProcessor;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Delete;
import com.sap.cds.ql.Insert;
import com.sap.cds.ql.Predicate;
import com.sap.cds.ql.RefSegment;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.StructuredType;
import com.sap.cds.ql.StructuredTypeRef;
import com.sap.cds.ql.Update;
import com.sap.cds.ql.cqn.AnalysisResult;
import com.sap.cds.ql.cqn.CqnDelete;
import com.sap.cds.ql.cqn.CqnInsert;
import com.sap.cds.ql.cqn.CqnModifier;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnSelectListItem;
import com.sap.cds.ql.cqn.CqnSelectListValue;
import com.sap.cds.ql.cqn.CqnSortSpecification;
import com.sap.cds.ql.cqn.CqnStatement;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.CqnUpdate;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.ql.cqn.ResolvedSegment;
import com.sap.cds.ql.impl.ExpressionVisitor;
import com.sap.cds.ql.impl.SelectBuilder;
import com.sap.cds.ql.impl.SelectListValueBuilder;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.CdsDeleteEventContext;
import com.sap.cds.services.cds.CdsReadEventContext;
import com.sap.cds.services.cds.CdsUpdateEventContext;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.draft.DraftAdministrativeData;
import com.sap.cds.services.draft.DraftCancelEventContext;
import com.sap.cds.services.draft.DraftCreateEventContext;
import com.sap.cds.services.draft.DraftEditEventContext;
import com.sap.cds.services.draft.DraftGcEventContext;
import com.sap.cds.services.draft.DraftNewEventContext;
import com.sap.cds.services.draft.DraftPatchEventContext;
import com.sap.cds.services.draft.DraftPrepareEventContext;
import com.sap.cds.services.draft.DraftSaveEventContext;
import com.sap.cds.services.draft.DraftService;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.After;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.cds.CapabilitiesHandler;
import com.sap.cds.services.impl.draft.ParentEntityLookup.ParentEntityLookupResult;
import com.sap.cds.services.impl.utils.CdsModelUtils;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.runtime.RequestContextRunner;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.DraftUtils;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.TenantAwareCache;
import com.sap.cds.services.utils.model.CdsAnnotations;
import com.sap.cds.util.CqnStatementUtils;
import com.sap.cds.util.OnConditionAnalyzer;

/**
 * An {@link EventHandler} to handle draft enabled entities.
 */
@ServiceName(value = "*", type = DraftService.class)
public class DraftHandler implements EventHandler {

	private static final Logger log = LoggerFactory.getLogger(DraftHandler.class);
	private static final String ADDED_COLUMN_PREFIX = "ADDED_FOR_DRAFT_SORTING_";

	private final TenantAwareCache<ParentEntityLookup, CdsModel> parentEntityLookups;

	public DraftHandler(CdsRuntime runtime) {
		parentEntityLookups = TenantAwareCache.create(() -> new ParentEntityLookup(RequestContext.getCurrent(runtime).getModel()), runtime);
	}

	// validate capabilities for draft events
	// we are not checking patch here, as this is used in the general new draft flow as well
	// also draft cancel is not checked for delete, as this might be a normal operation during new draft or edit draft flow as well
	@Before(event = DraftService.EVENT_DRAFT_NEW)
	@HandlerOrder(OrderConstants.Before.CHECK_CAPABILITIES)
	protected void checkCapabilityNewDraft(EventContext context) {
		if(!CapabilitiesHandler.getCapabilities(context).isInsertable()) {
			throw new ErrorStatusException(CdsErrorStatuses.ENTITY_NOT_INSERTABLE, context.getTarget().getQualifiedName());
		}
	}

	@Before(event = DraftService.EVENT_DRAFT_EDIT)
	@HandlerOrder(OrderConstants.Before.CHECK_CAPABILITIES)
	protected void checkCapabilityEditDraft(EventContext context) {
		if(!CapabilitiesHandler.getCapabilities(context).isUpdatable()) {
			throw new ErrorStatusException(CdsErrorStatuses.ENTITY_NOT_UPDATABLE, context.getTarget().getQualifiedName());
		}
	}

	@Before(event = {CqnService.EVENT_CREATE, CqnService.EVENT_UPSERT, CqnService.EVENT_UPDATE})
	@HandlerOrder(HandlerOrder.AFTER + OrderConstants.Before.FILTER_FIELDS)
	protected void clearDraftFields(EventContext context) {
		// draft fields should not be set manually
		CdsServiceUtils.getEntities(context).forEach(r -> {
			CdsModelUtils.visitDeep(context.getTarget(), r, (cdsEntity, data, parent, parentData) -> {
				data.remove(Drafts.HAS_DRAFT_ENTITY);
				data.remove(Drafts.HAS_ACTIVE_ENTITY);
				data.remove(Drafts.SIBLING_ENTITY);
				data.remove(Drafts.DRAFT_ADMINISTRATIVE_DATA);
				data.remove(DRAFT_ADMINISTRATIVE_DATA_DRAFT_UUID);
				if (parentData != null && !data.values().stream().anyMatch(e -> e != null)) {
					parentData.remove(parent.getName());
				}
			});
		});
	}

	@Before(event = {CqnService.EVENT_CREATE, CqnService.EVENT_UPSERT, CqnService.EVENT_UPDATE})
	// IsActiveEntity must be cleared late, as it is part of the key and relevant for validation targets
	@HandlerOrder(HandlerOrder.AFTER + OrderConstants.Before.VALIDATE_FIELDS)
	protected void clearIsActiveEntity(EventContext context) {
		CdsDataProcessor.create()
				.addConverter((p, e, t) -> e.getName().equals(Drafts.IS_ACTIVE_ENTITY), (p, e, v) -> Converter.REMOVE)
				.process(CdsServiceUtils.getEntities(context), context.getTarget());
	}

	@Before(service = "*", serviceType = PersistenceService.class, event = {CqnService.EVENT_CREATE, CqnService.EVENT_UPSERT, CqnService.EVENT_UPDATE})
	@HandlerOrder(HandlerOrder.AFTER + OrderConstants.Before.FILTER_FIELDS)
	protected void clearDraftFieldsOnPersistenceService(EventContext context) {
		if(DraftUtils.isDraftEnabled(context.getTarget()) && !context.getTarget().getQualifiedName().endsWith(CqnAdapter.DRAFT_SUFFIX)) {
			clearDraftFields(context);
		}
	}

	@Before(event = {EVENT_DRAFT_SAVE, EVENT_DRAFT_EDIT, EVENT_DRAFT_PREPARE})
	@HandlerOrder(OrderConstants.Before.CHECK_CAPABILITIES)
	protected void checkRootDraft(EventContext context) {
		if (!DraftUtils.isDraftEnabled(context.getTarget())) {
			throw new ErrorStatusException(CdsErrorStatuses.ENTITY_NOT_DRAFT_ENABLED, context.getTarget().getQualifiedName());
		}
		if (!DraftUtils.isDraftEnabledNoChild(context.getTarget())) {
			throw new ErrorStatusException(CdsErrorStatuses.ENTITY_NOT_ROOT, context.getTarget().getQualifiedName());
		}
	}

	@Before(event = {EVENT_DRAFT_CANCEL, EVENT_DRAFT_NEW, EVENT_DRAFT_PATCH})
	@HandlerOrder(OrderConstants.Before.CHECK_CAPABILITIES)
	protected void checkDraft(EventContext context) {
		if (!DraftUtils.isDraftEnabled(context.getTarget())) {
			throw new ErrorStatusException(CdsErrorStatuses.ENTITY_NOT_DRAFT_ENABLED, context.getTarget().getQualifiedName());
		}
	}

	@After
	protected void updateAdminDataOnNewDraft(DraftNewEventContext context) {
		if (DraftUtils.isDraftEnabled(context.getTarget()) && !DraftUtils.isDraftEnabledNoChild(context.getTarget())) {
			updateDraftAdministrativeDataChangeTime(context.getResult(), context);
		}
	}

	@Before
	protected void updateAdminDataOnDeletion(DraftCancelEventContext context) {
		if (DraftUtils.isDraftEnabled(context.getTarget()) && !DraftUtils.isDraftEnabledNoChild(context.getTarget())) {
			CqnSelect select = CqnAdapter.toSelect(context.getCqn()).columns(DRAFT_ADMINISTRATIVE_DATA_DRAFT_UUID);
			// ensure to only select inactive entities
			select = CQL.copy(select, new CqnModifier(){
				@Override
				public Predicate where(Predicate where) {
					Predicate isActiveEntityFalse = CQL.get(Drafts.IS_ACTIVE_ENTITY).eq(false);
					if (where == null) {
						return isActiveEntityFalse;
					}
					return CQL.and(where, isActiveEntityFalse);
				}
			});
			updateDraftAdministrativeDataChangeTime(read(select, context), context);
		}
	}

	private void updateDraftAdministrativeDataChangeTime(Result result, EventContext context) {
		Instant now = Instant.now();
		List<Map<String, Object>> updates = result.stream()
										.map(row -> row.get(DRAFT_ADMINISTRATIVE_DATA_DRAFT_UUID))
										.distinct()
										.map(uuid -> ImmutableMap.of(DRAFT_UUID, uuid, LAST_CHANGE_DATE_TIME, now))
										.collect(Collectors.toList());
		if (!updates.isEmpty()) {
			PersistenceService persistenceService = context.getCdsRuntime().getServiceCatalog().getService(PersistenceService.class, PersistenceService.DEFAULT_NAME);
			persistenceService.run(Update.entity(DraftAdministrativeData.CDS_NAME).entries(updates));
		}
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected Result defaultNew(DraftNewEventContext context) {
		// Add draft fields to each entity
		CqnInsert insert = Insert.copy(context.getCqn());
		insert.entries().stream().forEach(m -> addDraftFields(m, false, context.getTarget(), context, context.getCqn().ref()));

		Result result = ((DraftServiceImpl) context.getService()).createDraft(insert);
		if (!DraftUtils.isDraftEnabledNoChild(context.getTarget())) {
			// check if the entity has a parent entity in draft mode

			List<ParentEntityLookupResult> parentLookups = parentEntityLookups.findOrCreate().lookupParent(context.getTarget());
			if (parentLookups == null || parentLookups.isEmpty()) {
				throw new ErrorStatusException(CdsErrorStatuses.NO_PARENT_ENTITY, context.getTarget().getQualifiedName());
			}
			result.forEach(r -> {
				int numberParents = 0;
				for (ParentEntityLookupResult parentLookup: parentLookups) {
					if (DraftUtils.isDraftEnabled(parentLookup.getParentEntity())) {
						String compositionName = parentLookup.getComposition().getName();
						CqnPredicate pred = CQL.get(Drafts.IS_ACTIVE_ENTITY).eq(false);
						for (Map.Entry<String, Object> key: getKeys(r, context.getTarget()).entrySet()) {
							CqnPredicate tmpPred = CQL.to(compositionName).get(key.getKey()).eq(key.getValue());
							pred = CQL.and(pred, tmpPred);
						}
						CqnSelect select = Select.from(parentLookup.getParentEntity()).where(pred);
						Result result2 = read(select, context);
						if (result2.first().isPresent()) {
							++numberParents;
						}
					}
				}
				if (numberParents == 0) {
					throw new ErrorStatusException(CdsErrorStatuses.PARENT_NOT_EXISTING, context.getTarget().getQualifiedName());
				}
				if (numberParents > 1) {
					throw new ErrorStatusException(CdsErrorStatuses.MULTIPLE_PARENTS, numberParents, context.getTarget().getQualifiedName());
				}
			});
		}
		return result;
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected Result defaultSave(DraftSaveEventContext context) {
		// Get entities to save
		CqnSelect select = Select.copy(context.getCqn())
				.columns(expandCompositions(context.getTarget()));

		Result toSave = read(select, context);

		// restore null values in the result, because they are not returned by cds4j
		CdsDataProcessor.create().addGenerator((p, e, t) -> t.isSimple(), (p, e, n) -> null) //
		.process(toSave, context.getTarget());

		if (toSave.rowCount() == 0) {
			throw new ErrorStatusException(CdsErrorStatuses.ENTITY_INSTANCE_NOT_FOUND,
					context.getTarget().getQualifiedName(),
					com.sap.cds.services.utils.model.CdsModelUtils.getTargetKeysAsString(context.getModel(), select));
		}

		List<Row> newEntities = new ArrayList<>();
		List<Row> existingEntities = new ArrayList<>();
		toSave.forEach(r -> {
			if (!Boolean.FALSE.equals(r.get(Drafts.IS_ACTIVE_ENTITY))) {
				throw new ErrorStatusException(CdsErrorStatuses.ENTITY_NOT_INACTIVE);
			}
			if ((Boolean) r.get(Drafts.HAS_ACTIVE_ENTITY)) {
				existingEntities.add(r);
			} else {
				newEntities.add(r);
			}
		});

		// delete the draft entities
		CqnDelete delete = CqnAdapter.toDelete(select);
		Result deleteResult = cancel(delete, context, context.getCqnNamedValues() != null ? Arrays.asList(context.getCqnNamedValues()): Collections.emptyList());
		if (deleteResult.rowCount() != toSave.rowCount()) {
			throw new ErrorStatusException(CdsErrorStatuses.ENTITY_NOT_INACTIVE);
		}

		Result result = null;
		if (!newEntities.isEmpty()) {
			// insert new active entities
			CqnInsert insert = Insert.into(context.getTarget()).entries(newEntities);
			result = context.getService().run(insert);
		}
		if (!existingEntities.isEmpty()) {
			// update the active entities
			CqnUpdate update = Update.entity(context.getTarget()).entries(existingEntities);
			result = mergeInsertResults(result, context.getService().run(update));
		}
		adaptSaveResult(result, context.getTarget()); // NOSONAR
		return result;
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected Result defaultEdit(DraftEditEventContext context) {
		// Get active entities
		CqnSelect select = Select.copy(context.getCqn())
				.columns(expandCompositions(context.getTarget()));

		// ensure that we read the data for edit without a locale!
		Result toEdit = context.getCdsRuntime().requestContext().modifyParameters(param -> {
			param.setLocale(null);
		}).run( reqContext -> {
			return read(select, context);
		});

		if (toEdit.rowCount() == 0) {
			throw new ErrorStatusException(CdsErrorStatuses.ENTITY_INSTANCE_NOT_FOUND,
					context.getTarget().getQualifiedName(),
					com.sap.cds.services.utils.model.CdsModelUtils.getTargetKeysAsString(context.getModel(), select));
		}
		toEdit.forEach(r -> {
			if (!Boolean.TRUE.equals(r.get(Drafts.IS_ACTIVE_ENTITY))) {
				throw new ErrorStatusException(CdsErrorStatuses.ENTITY_NOT_ACTIVE);
			}
			// clear on insert and on update fields, to ensure they are set after edit to a new value
			clearOnUpdateFields(r, context.getTarget());
			addDraftFields(r, true, context.getTarget(), context, context.getCqn().ref());
		});

		// Cancel existing draft
		if (context.getPreserveChanges() != null && !context.getPreserveChanges()) {

			CqnDelete delete = CqnAdapter.toDelete(context.getCqn());
			delete = CqnAdapter.replaceIsActiveEntity(delete, true);
			delete = addTimeoutConstraint(delete, context);

			final CqnDelete deleteStatement = delete;
			privileged(context).run((requestContext) -> {
				context.getService().cancelDraft(deleteStatement);
			});
		}

		// Insert the entities into the drafts table
		CqnInsert insert = Insert.into(context.getTarget().getQualifiedName()).entries(toEdit);
		Result result = ((DraftServiceImpl)context.getService()).createDraft(insert);
		return result;
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected Result defaultPatch(DraftPatchEventContext context) {
		CqnUpdate update = context.getCqn();
		update = addSecurityConstraints(update, context);
		List<CqnUpdate> statements = removeActiveTargets(CqnAdapter.adapt(update, context), context);
		return execute(statements, context.getCqnValueSets(), context);
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected Result defaultPrepare(DraftPrepareEventContext context) {
		Result result = context.getService().run(context.getCqn());
		if (result.rowCount() == 0) {
			throw new ErrorStatusException(CdsErrorStatuses.ENTITY_INSTANCE_NOT_FOUND,
					context.getTarget().getQualifiedName(), com.sap.cds.services.utils.model.CdsModelUtils
					.getTargetKeysAsString(context.getModel(), context.getCqn()));
		}
		// Just some basic check. The actual implementation is done by the application
		result.forEach(r -> {
			if (!Boolean.FALSE.equals(r.get(Drafts.IS_ACTIVE_ENTITY))) {
				throw new ErrorStatusException(CdsErrorStatuses.ENTITY_NOT_INACTIVE);
			}
		});
		return result;
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected Result defaultCancel(DraftCancelEventContext context) {
		return cancel(context.getCqn(), context, context.getCqnValueSets());
	}

	@On
	@HandlerOrder(OrderConstants.On.FEATURE)
	protected void defaultRead(CdsReadEventContext context) {
		context.setResult(read(context.getCqn(), context.getCqnNamedValues(), context));
	}

	@On
	@HandlerOrder(OrderConstants.On.FEATURE)
	protected void defaultUpdate(CdsUpdateEventContext context) {
		if (DraftUtils.isDraftEnabled(context.getTarget())) {
			CqnUpdate update = context.getCqn();
			AnalysisResult entityPath = CdsModelUtils.getEntityPath(context.getCqn(), context.getModel());
			CqnUpdate updateActive = addLockingConstraints(Update.copy(update), context, entityPath);
			List<CqnUpdate> stmts = removeDraftTargets(CqnAdapter.adapt(updateActive, context), context);

			context.setResult(execute(stmts, context.getCqnValueSets(), context));
		}
	}

	@On
	@HandlerOrder(OrderConstants.On.FEATURE)
	protected void defaultDelete(CdsDeleteEventContext context) {
		if (DraftUtils.isDraftEnabled(context.getTarget())) {
			// If the entity is draft enabled we have to adapt the where condition
			// in case draft fields are referenced
			DraftService service = (DraftService) context.getService();
			// this does nothing if only inactive entities should be deleted
			Result result = execute(removeDraftTargets(CqnAdapter.adapt(context.getCqn(), context), context), context, context.getCqnValueSets());
			Result resultDraft = null;
			// If active entities were deleted, execute the adapted statement again
			// on the draft table to also delete the corresponding draft entities
			// TODO: Use cascade.delete annotation once CDS4J is resolving the Projections
			if (result.rowCount() > 0) {
				// TODO this is not working if draft fields other than IsActiveEntity are referenced, the only solution I see is to execute a read first
				for (CqnDelete d: CqnAdapter.adaptActiveEntity(context.getCqn(), context)) {
					// since the active entity was deleted, also delete the draft
					Result deleteResult = privileged(context).run((requestContext) -> {
						return service.cancelDraft(d);
					});

					long[] deleteCount = addRowCounts(deleteResult, resultDraft);
					resultDraft = ResultBuilder.deletedRows(deleteCount).result();
				}
			}
			long[] deleteCount = addRowCounts(result, resultDraft);
			result = ResultBuilder.deletedRows(deleteCount).result();
			context.setResult(result);
		}
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected Result createDraft(DraftCreateEventContext context) {
		List<CqnInsert> inserts = CqnAdapter.adapt(context.getCqn(), context);
		removeActiveTargets(inserts, context);
		if (inserts.size() > 1) {
			throw new ErrorStatusException(CdsErrorStatuses.MISSING_ISACTIVEENTITY_KEY);
		}
		return CdsServiceUtils.getDefaultPersistenceService(context).run(inserts.get(0));
	}

	@After(event = { CqnService.EVENT_CREATE, CqnService.EVENT_UPSERT, CqnService.EVENT_UPDATE })
	@HandlerOrder(OrderConstants.After.ADD_FIELDS)
	@SuppressWarnings("unchecked")
	protected void addActiveDraftFields(EventContext context) {
		List<?> result = CdsServiceUtils.getResult(context).list();
		CdsModelUtils.visitDeep(context.getTarget(), (List<Map<String, Object>>) result, (cdsEntity, data, parent) -> {
			if(DraftUtils.isDraftEnabled(cdsEntity)) {
				data.forEach(m -> {
					m.put(Drafts.IS_ACTIVE_ENTITY, true);
					m.put(Drafts.HAS_ACTIVE_ENTITY, false);
					// TODO might also have a draft entity, but we should prevent this
					m.put(Drafts.HAS_DRAFT_ENTITY, false);
				});
			}
		});
	}

	@SuppressWarnings("unchecked")
	@Before(event = DraftService.EVENT_DRAFT_PATCH)
	protected void updateDraftAdministrativeData(DraftPatchEventContext context) {
		List<Map<String, Object>> entries = context.getCqn().entries();
		// TODO should we be able to get the NOW time stamp from the context?
		Instant now = Instant.now();
		CdsModelUtils.visitDeep(context.getTarget(), entries, (entity, data, parent) -> {
			if (entity.findElement(Drafts.DRAFT_ADMINISTRATIVE_DATA).isPresent()) {
				if (!entity.getQualifiedName().equals(context.getTarget().getQualifiedName())) {
					// deep updates on draft entities are not supported
					throw new ErrorStatusException(CdsErrorStatuses.DRAFT_DEEP_UPDATE);
				}
				data.forEach(row -> {
					DraftAdministrativeData adminData;
					if (row.containsKey(Drafts.DRAFT_ADMINISTRATIVE_DATA)) {
						adminData = Struct.access((Map<String, Object>) row.get(Drafts.DRAFT_ADMINISTRATIVE_DATA))
								.as(DraftAdministrativeData.class);
					} else {
						adminData = Struct.create(DraftAdministrativeData.class);
						row.put(Drafts.DRAFT_ADMINISTRATIVE_DATA, adminData);
					}
					adminData.setLastChangeDateTime(now);
				});
			}
		});
	}

	private RequestContextRunner privileged(EventContext context) {
		return context.getCdsRuntime().requestContext().privilegedUser();
	}

	private CqnUpdate addSecurityConstraints(CqnUpdate update, EventContext context) {
		CqnPredicate predicate = CqnAdapter.getSecurityConstraints(context);
		return modifiedWhere(update, andPredicate(predicate));
	}

	private CqnDelete addSecurityConstraints(CqnDelete delete, EventContext context) {
		CqnPredicate predicate = CqnAdapter.getSecurityConstraints(context);
		return modifiedWhere(delete, andPredicate(predicate));
	}

	private CqnUpdate addLockingConstraints(CqnUpdate update, EventContext context, AnalysisResult entityPath) {
		if (!context.getCdsRuntime().getEnvironment().getCdsProperties().getSecurity().getDraftProtection().isEnabled()) {
			return update;
		}
		// TODO this should work once cds4j supports paths in update, for now we work with a sub query
		//			predicate = predicate.or(CQL.get(Drafts.IS_ACTIVE_ENTITY).eq(true).and(CQL.get(Drafts.HAS_DRAFT_ENTITY).eq(false)));
		String draftEntity = CqnAdapter.getDraftsEntity(entityPath.targetEntity().getQualifiedName());
		List<CdsElement> keys = entityPath.targetEntity().keyElements().filter(e -> !e.getType().isAssociation() && !e.isVirtual() && !e.getName().equals(Drafts.IS_ACTIVE_ENTITY)).collect(Collectors.toList());
		CqnPredicate predicate = Select.from("").where(c -> c.exists(o -> Select.from(draftEntity).where(i -> createExistsDraftCondition(o, i, keys))).not()).where().get();

		return modifiedWhere(update, andPredicate(predicate));
	}

	private static CqnPredicate createExistsDraftCondition(StructuredType<?> outer, StructuredType<?> inner, List<CdsElement> keys) {
		List<CqnPredicate> predicates = new ArrayList<>(keys.size());
		for (CdsElement key: keys) {
			predicates.add(outer.get(key.getName()).eq(inner.get(key.getName())));
		}
		return CQL.and(predicates);
	}

	private CqnDelete addTimeoutConstraint(CqnDelete delete, EventContext context) {
		Instant timeoutThreshold = DraftModifier.getCancellationThreshold(context);
		CqnPredicate predicate = Select.from("").where(c -> c.exists(b -> Select.from(DraftAdministrativeData.CDS_NAME).where(e -> e.get(DRAFT_UUID).eq(b.get(DRAFT_ADMINISTRATIVE_DATA_DRAFT_UUID))
				.and(e.get(LAST_CHANGE_DATE_TIME).le(timeoutThreshold))))).where().get();
		//		pred = CQL.exists(Select.from(DraftAdministrativeData.QUALIFIED_ENTITY_NAME).where(e -> e.get(DraftAdministrativeData.DRAFT_UUID).eq(b.get(Drafts.DRAFT_ADMINISTRATIVE_DATA_DRAFT_UUID))
		//				.and(e.get(DraftAdministrativeData.LAST_CHANGE_DATE_TIME).le(getTimeoutThreshold()))));
		return modifiedWhere(delete, andPredicate(predicate));
	}

	private Result cancel(CqnDelete delete, EventContext context, Iterable<Map<String, Object>> valueSets) {
		delete = addSecurityConstraints(delete, context);
		List<CqnDelete> statements = removeActiveTargets(CqnAdapter.adapt(delete, context), context);
		return execute(statements, context, valueSets);
	}

	private <T extends CqnStatement> List<T> removeActiveTargets(List<T> statements, EventContext context) {
		statements.removeIf(d -> !CdsModelUtils.getEntityPath(d.ref(), context.getModel()).targetEntity().getQualifiedName().endsWith(CqnAdapter.DRAFT_SUFFIX));
		return statements;
	}

	private <T extends CqnStatement> List<T> removeDraftTargets(List<T> statements, EventContext context) {
		statements.removeIf(d -> CdsModelUtils.getEntityPath(d.ref(), context.getModel()).targetEntity().getQualifiedName().endsWith(CqnAdapter.DRAFT_SUFFIX));
		return statements;
	}

	private Result read(CqnSelect select, Map<String, Object> cqnNamedValues, EventContext context) {

		// for draft enabled entities we request both tables with the same statement and merge the result

		Map<String, String> jsonToDisplayName = addMissingItems(context.getModel(), select, select.orderBy());
		List<CqnSelect> statements = CqnAdapter.adapt(select, context);
		Result result = execute(select, statements, context, cqnNamedValues, jsonToDisplayName);
		removeAddedItems(jsonToDisplayName, result);
		result = setRowType(select, context, result);

		return result;
	}

	private Result setRowType(CqnSelect select, EventContext context, Result result) {
		CdsStructuredType rowType = CqnStatementUtils.rowType(context.getModel(), select);
		return ResultBuilder.selectedRows(result.list()).inlineCount(result.inlineCount()).rowType(rowType).result();
	}

	private Result execute(List<CqnDelete> statements, EventContext context, Iterable<Map<String, Object>> valueSets) {
		return execute((d, service) -> service.run((CqnDelete) d, valueSets), statements, context);
	}

	private Result execute(List<CqnUpdate> statements, Iterable<Map<String, Object>> valueSets, EventContext context) {
		return execute((u, service) -> service.run((CqnUpdate) u, valueSets), statements, context);
	}

	private Result execute(BiFunction<CqnStatement, PersistenceService, Result> executor, List<? extends CqnStatement> statements, EventContext context) {
		PersistenceService service = CdsServiceUtils.getDefaultPersistenceService(context);
		List<Map<String, Object>> rows = new ArrayList<>();
		List<Long> rowCount = new ArrayList<>();
		for (CqnStatement statement: statements) {
			Result result = executor.apply(statement, service);
			if (rows.isEmpty()) {
				rows.addAll(result.list());
			} else {
				List<Row> resultList = result.list();
				for (int i=0; i<resultList.size(); ++i) {
					Row row = resultList.get(i);
					if (!row.isEmpty()) {
						rows.set(i, row);
					}
				}
			}
			for (int i = 0; i < result.batchCount(); ++i) {
				if (rowCount.size() > i) {
					rowCount.set(i, rowCount.get(i) + result.rowCount(i));
				} else {
					rowCount.add(result.rowCount(i));
				}
			}
		}
		long[] rowCountArray = new long[rowCount.size()];
		for (int i = 0; i < rowCount.size(); ++i) {
			rowCountArray[i] = rowCount.get(i);
		}
		if (rows.isEmpty()) {
			return ResultBuilder.deletedRows(rowCountArray).result();
		}
		ResultBuilder resultBuilder = ResultBuilder.batchUpdate();
		for (int i = 0; i < rowCountArray.length; ++i) {
			resultBuilder.addUpdatedRows(rowCountArray[i], rows.get(i));
		}
		return resultBuilder.result();
	}

	private Result execute(CqnSelect select, List<CqnSelect> statements, EventContext context, Map<String, Object> cqnNamedValues, Map<String, String> jsonToDisplayName) {
		if (!select.orderBy().isEmpty() || select.hasLimit()) {
			// select sorted columns
			List<CqnSelect> draftStatements = new ArrayList<>(statements);
			List<CqnSelect> activeStatements = new ArrayList<>(statements);
			removeActiveTargets(draftStatements, context);
			removeDraftTargets(activeStatements, context);
			// we assume a low number of draft entities
			draftStatements.forEach(s -> ((SelectBuilder<?>)s).limit(-1, 0));
			Result draftResult = executePlain(draftStatements, context, cqnNamedValues);
			long numDrafts = draftResult.rowCount();
			long skipDiff = 0;
			long oldTop = 0;
			if (select.hasLimit()) {
				// we have to select more active entities to sort in the draft entities correctly
				long oldSkip = select.skip();
				oldTop = select.top();
				long newSkip = Math.max(0, oldSkip - numDrafts);
				// handle overflow
				long newTop = Math.max(oldTop + numDrafts, oldTop);
				skipDiff = oldSkip - newSkip;
				activeStatements.forEach(s -> ((Select<?>)s).limit(newTop, newSkip));
			}
			Result activeResult = executePlain(activeStatements, context, cqnNamedValues);
			List<Row> orderedResult = sortResults(context, activeResult, draftResult, select.orderBy(), jsonToDisplayName);
			if (select.hasLimit()) {
				long newEnd = Math.max(oldTop + skipDiff, oldTop);
				long end = Math.min(newEnd, orderedResult.size());
				orderedResult = orderedResult.subList((int) skipDiff, (int) end);
			}
			return ResultBuilder.selectedRows(orderedResult).inlineCount(calcInlineCount(draftResult, activeResult)).result();
		}
		return executePlain(statements, context, cqnNamedValues);
	}

	private static long calcInlineCount(Result draftResult, Result activeResult) {
		long inlineCount = activeResult.inlineCount();
		if (inlineCount < 0) {
			inlineCount = draftResult.inlineCount();
		} else if (draftResult.inlineCount() >= 0) {
			inlineCount += draftResult.inlineCount();
		}
		return inlineCount;
	}

	private static void removeAddedItems(Map<String, String> jsonToDisplayName, Result orderedResult) {
		// TODO - make this more strict ?
		Set<String> added = jsonToDisplayName.values().stream().filter(v -> v.startsWith(ADDED_COLUMN_PREFIX)).collect(toSet());
		removeAddedItems(orderedResult, added);
	}

	private static void removeAddedItems(Result orderedResult, Set<String> added) {
		if (!added.isEmpty()) {
			// Generated aliases are removed from the result set -> they have TRUE in the value pair
			orderedResult.forEach(row -> row.keySet().removeAll(added));
		}
	}

	/**
	 * Adds select list items for values on the order by list, which are not present
	 * on the select list
	 * 
	 * @return map of the JSON representation of the select list values (without
	 *         alias) to the display name
	 */
	private Map<String, String> addMissingItems(CdsModel model, CqnSelect select,
			List<CqnSortSpecification> sortSpecifications) {
		List<CqnSelectListValue> missingSLVs = new ArrayList<>(sortSpecifications.size());

		/*
		 * Select list items required for sorting will be added. They will have a
		 * generated alias if the value is not yet present on the select list.
		 */
		Map<String, String> jsonToDisplayName;
		if (!select.orderBy().isEmpty()) {
			jsonToDisplayName = mapSelectListValueJsonToDisplayNames(model, select);
			int i = 0;
			for (CqnSortSpecification sortSpec : sortSpecifications) {
				CqnValue val = sortSpec.value();
				String json = json(val);
				String displayName = jsonToDisplayName.get(json);
				if (displayName == null) {
					displayName = ADDED_COLUMN_PREFIX + (++i);
					CqnSelectListValue slv = SelectListValueBuilder.select(val).as(displayName).build();
					missingSLVs.add(slv);
					jsonToDisplayName.put(json, displayName);
				}
			}
			if (!missingSLVs.isEmpty()) {
				List<CqnSelectListItem> oldItems = select.items().isEmpty() ? Collections.singletonList(CQL.star()) : select.items();
				List<CqnSelectListItem> newItems = new ArrayList<>(oldItems.size() + missingSLVs.size());
				newItems.addAll(oldItems);
				newItems.addAll(missingSLVs);
				((Select<?>) select).columns(newItems);
			}
			return jsonToDisplayName;
		}
		return Collections.emptyMap();
	}

	private static String json(CqnValue val) {
		if (val.isRef()) {
			val = CQL.get(val.asRef().segments());
		}
		return val.toJson();
	}

	private static Map<String, String> mapSelectListValueJsonToDisplayNames(CdsModel model, CqnSelect select) {
		select = resolveStar(select, entity(model, select.from().asRef()));
		// Certain combination of expands may lead to same items being in the select -> the collector will merge them
		return select.items().stream().flatMap(CqnSelectListItem::ofValue)
				.collect(toMap(slv -> json(slv.value()), CqnSelectListValue::displayName, (s, s2) -> s2));
	}

	private static String getItemName(CqnSortSpecification spec, Map<String, String> jsonToDisplayName) {
		return jsonToDisplayName.get(json(spec.value()));
	}

	@SuppressWarnings({ "unchecked", "rawtypes" })
	private static List<Row> sortResults(EventContext context, Result active, Result draft, List<CqnSortSpecification> sortSpecifications, Map<String, String> jsonToDisplayName) {
		CdsEntity entity = context.getTarget();
		ArrayList<Row> result = new ArrayList<Row>((int) (active.rowCount() + draft.rowCount()));
		Iterator<Row> activeIter = active.iterator();
		Iterator<Row> draftIter = draft.iterator();
		Row rActive = activeIter.hasNext() ? activeIter.next() : null;
		Row rDraft = draftIter.hasNext() ? draftIter.next() : null;
		while (rActive != null || rDraft != null) {
			int comparison = Objects.compare(rDraft, rActive, (o1, o2) -> {
				if (o1 == null) {
					return 1;
				}
				if (o2 == null) {
					return -1;
				}
				int comp;
				for (CqnSortSpecification spec: sortSpecifications) {
					String item = getItemName(spec, jsonToDisplayName);
					int orderFactor = spec.order() == CqnSortSpecification.Order.DESC ? -1 : 1;
					Object value1 = o1.getPath(item);
					Object value2 = o2.getPath(item);
					if (value1 == null && value2 == null) {
						continue;
					}
					// put null values always at the beginning of the list (if ascending order) to maintain DB sort order
					if (value1 == null) {
						return -orderFactor;
					}
					if (value2 == null) {
						return orderFactor;
					}
					if (!(value1 instanceof Comparable) || !(value2 instanceof Comparable)) {
						throw new ErrorStatusException(CdsErrorStatuses.INVALID_SORT_ELEMENT, item, entity.getQualifiedName());
					}
					Locale locale = context.getParameterInfo().getLocale();
					if ((value1 instanceof String) && (value2 instanceof String) && locale != null) {
						Collator collator = Collator.getInstance(locale);
						comp = collator.compare(value1, value2);
					} else {
						comp = ((Comparable)value1).compareTo(value2);
					}

					if (comp != 0) {
						return orderFactor * comp;
					}
				}
				// sort active and inactive entities always in the same order
				if (o1.containsKey(Drafts.IS_ACTIVE_ENTITY) && (Boolean) o1.get(Drafts.IS_ACTIVE_ENTITY)) {
					if (o2.containsKey(Drafts.IS_ACTIVE_ENTITY) && (Boolean)o2.get(Drafts.IS_ACTIVE_ENTITY)) {
						return 0;
					}
					return 1;
				} else {
					if (o2.containsKey(Drafts.IS_ACTIVE_ENTITY) && (Boolean)o2.get(Drafts.IS_ACTIVE_ENTITY)) {
						return -1;
					}
					return 0;
				}
			});
			if (comparison > 0) {
				result.add(rActive);
				if (activeIter.hasNext()) {
					rActive = activeIter.next();
				} else {
					rActive = null;
				}
			} else {
				result.add(rDraft);
				if (draftIter.hasNext()) {
					rDraft = draftIter.next();
				} else {
					rDraft = null;
				}
			}
		}
		return result;
	}

	private Result executePlain(List<CqnSelect> statements, EventContext context, Map<String, Object> cqnNamedValues) {
		PersistenceService service = CdsServiceUtils.getDefaultPersistenceService(context);
		Iterable<Row> result = null;
		long inlineCount = -1;
		Result tmpResult;
		for (CqnSelect select: statements) {
			tmpResult = service.run(select, cqnNamedValues);
			if (result == null) {
				result = tmpResult;
			} else {
				result = Iterables.concat(result, tmpResult);
			}
			if (tmpResult.inlineCount() >= 0) {
				if (inlineCount < 0) {
					inlineCount = tmpResult.inlineCount();
				} else {
					inlineCount += tmpResult.inlineCount();
				}
			}
		}
		if (result == null) {
			return ResultBuilder.selectedRows(Collections.emptyList()).result();
		}
		List<Row> resultList = Lists.newArrayList(result);
		mergeExpands(resultList, context);
		return ResultBuilder.selectedRows(resultList).inlineCount(inlineCount).result();
	}

	@SuppressWarnings({ "unchecked", "rawtypes" })
	private static void mergeExpands(List<Row> resultList, EventContext context) {
		DataProcessor.create().bulkAction((entity, data) -> {
			Map<String, String> draftAssocs = entity.associations()
					.map(a -> a.getName())
					.filter(n -> n.endsWith(CqnAdapter.DRAFT_SUFFIX))
					.collect(Collectors.toMap(n -> n, n -> n.substring(0, n.length() - CqnAdapter.DRAFT_SUFFIX.length())));

			for(Map<String, Object> row : data) {
				for(Entry<String, String> assoc : draftAssocs.entrySet()) {
					String draftAssocName = assoc.getKey();
					Object draftAssocValue = row.get(draftAssocName);
					if (draftAssocValue != null) {
						String assocName = assoc.getValue();
						Object assocValue = row.get(assocName);
						if (draftAssocValue instanceof List) {
							if (assocValue != null) {
								((List) assocValue).addAll((List) draftAssocValue);
							} else {
								row.put(assocName, draftAssocValue);
							}
						} else if (assocValue == null) {
							// override the non draft result
							row.put(assocName, draftAssocValue);
						}
					}
					row.remove(draftAssocName);
				}
			}
		}).process(resultList, context.getTarget());
	}

	private Result read(CqnSelect select, EventContext context) {
		return read(select, Collections.emptyMap(), context);
	}

	private long[] addRowCounts(Result... results) {
		long[] result = new long[0];
		if (results.length == 0) {
			return result;
		}
		for (Result r: results) {
			if (r != null) {
				if (result.length == 0) {
					result = new long[r.batchCount()];
				}
				for (int i = 0; i < result.length; ++i) {
					result[i] += r.rowCount(i);
				}
			}
		}
		return result;
	}

	/**
	 * Returns a map containing the keys of {@code entity} retrieved from {@code m}.
	 * @param m the map containing the keys
	 * @param entity the entity
	 * @return a map containing the keys of {@code entity}
	 * @throws ErrorStatusException if {@code m} does not contain all keys
	 */
	private static Map<String, Object> getKeys(Map<String, Object> m, CdsEntity entity) {
		Map<String, Object> result = new HashMap<>(m);
		result.entrySet().removeIf(entry -> {
			return !entity.findElement(entry.getKey()).map(e -> e.isKey()).orElse(false)
					&& !Drafts.IS_ACTIVE_ENTITY.equals((entry.getKey()));

		});
		if (result.size() != entity.keyElements().filter(element -> !element.getType().isAssociation()).count()) {
			log.error("Failed to find keys of {} in result {}", entity, m);
			throw new ErrorStatusException(CdsErrorStatuses.NO_KEYS_IN_RESULT, entity);
		}
		return result;
	}

	private static void clearOnUpdateFields(Map<String, Object> m, CdsEntity entity) {
		CdsModelUtils.visitDeep(entity, m, (cdsEntity, data, parent, parentData) -> {
			cdsEntity.elements()
			.filter(e -> CdsAnnotations.ON_UPDATE.getOrDefault(e) != null)
			.map(e -> e.getName()).forEach(e -> data.remove(e));
		});
	}

	@SuppressWarnings("unchecked")
	private static <T> Function<T, CqnSelectListItem>[] expandCompositions(CdsEntity entity) {
		List<Function<StructuredType<?>, CqnSelectListItem>> columns = new ArrayList<>();
		columns.add(c -> c._all());
		entity.compositions().forEach(co -> columns.add(c -> c.to(co.getName()).expand(expandCompositions(entity.getTargetOf(co.getName())))));
		return columns.toArray(new Function[columns.size()]);
	}


	/**
	 * Adds the draft fields to an entity.
	 * @param m the entity
	 * @param hasActiveEntity should be {@code true} if the entity has an active entity, {@code false} otherwise
	 */
	private void addDraftFields(Map<String, Object> m, boolean hasActiveEntity, CdsEntity entity, EventContext context, CqnStructuredTypeRef ref) {
		final String draftUUID = determineDraftUuid(m, entity, context, ref);
		CdsModelUtils.visitDeep(entity, m, (cdsEntity, mapData, parent, parentData) -> {
			if (parent != null && !parent.getType().as(CdsAssociationType.class).isComposition()) {
				return;
			}
			Drafts draft = Struct.access(mapData).as(Drafts.class);
			draft.setHasActiveEntity(hasActiveEntity);
			draft.setHasDraftEntity(false);
			draft.setIsActiveEntity(false);
			draft.setDraftAdministrativeDataDraftUuid(draftUUID);
			if (DraftUtils.isDraftEnabledNoChild(cdsEntity)) {
				DraftAdministrativeData data = Struct.create(DraftAdministrativeData.class);
				String userName = context.getUserInfo().getName();
				data.setCreatedByUser(userName);
				data.setDraftUUID(draftUUID);
				// copied from the ManagedAspectHandler
				// TODO should we be able to get the NOW time stamp from the context?
				Instant now = Instant.now();
				data.setCreationDateTime(now);
				data.setLastChangeDateTime(now);
				data.setInProcessByUser(userName);
				data.setLastChangedByUser(userName);
				draft.setDraftAdministrativeData(data);
			}
		});
	}

	private String determineDraftUuid(Map<String, Object> m, CdsEntity entity, EventContext context, CqnStructuredTypeRef ref) {
		Drafts draftData = Struct.access(m).as(Drafts.class);
		if (draftData.getDraftAdministrativeDataDraftUuid() != null) {
			// for forward linked associations we cannot determine the draftUUID from the data
			// if the foreign key is not set in the parent entity, therefore it must be provided
			return draftData.getDraftAdministrativeDataDraftUuid();
		} else if (DraftUtils.isDraftEnabledNoChild(entity)) {
			// root entity -> we generate DraftAdministrativeData
			return UUID.randomUUID().toString();
		}
		// child entity -> we need to find out the parent entity to retrieve the draftUUID
		Result result = null;
		long numParents = 0;
		Set<String> uniqueParents = new HashSet<>();
		List<ParentEntityLookupResult> lookupResultList = parentEntityLookups.findOrCreate().lookupParent(entity);
		AnalysisResult analysisResult = CdsModelUtils.getEntityPath(ref, context.getModel());
		Iterator<ResolvedSegment> iter = analysisResult.reverse();
		iter.next();
		CdsEntity potentialParent = iter.hasNext() ? iter.next().entity() : null;
		if (potentialParent != null && lookupResultList.stream().anyMatch(lr -> lr.getParentEntity().equals(potentialParent))) {
			// we have a path expression where the parent entity comes before the last segment -> we get the parent by omitting the last segment
			StructuredTypeRef newRef = ExpressionVisitor.copy(ref, new CqnModifier() {
				@Override
				public CqnStructuredTypeRef ref(StructuredTypeRef ref) {
					List<RefSegment> segmentsToParent = ref.segments().subList(0, ref.segments().size() - 1);
					RefSegment lastSegment = segmentsToParent.get(segmentsToParent.size() - 1);
					// ensure that the inactive entity is selected
					Predicate inactive = CQL.get(Drafts.IS_ACTIVE_ENTITY).eq(false);
					CqnPredicate newPred = lastSegment.filter().map(f -> CQL.and(f, inactive)).orElse(inactive);
					lastSegment.filter(newPred);
					return CQL.to(segmentsToParent).asRef();
				}
			});
			result = read(Select.from(newRef).columns(DRAFT_ADMINISTRATIVE_DATA_DRAFT_UUID), context);
			numParents = result.rowCount();
		} else {
			// the child is the root entity of the statement -> we need to find the parent through a lookup
			Result tmpResult;
			for (ParentEntityLookupResult lookupResult: lookupResultList) {
				CdsEntity parentEntity = lookupResult.getParentEntity();

				if (!uniqueParents.contains(parentEntity.getQualifiedName()) && DraftUtils.isDraftEnabled(parentEntity)) {
					uniqueParents.add(parentEntity.getQualifiedName());

					OnConditionAnalyzer analyzer = new OnConditionAnalyzer(lookupResult.getComposition(), false);

					// m is the whole entry map of the child insert, but
					// analyzer.getFkValues(m) only returns the foreign key of the child that refers to the parent
					Map<String, Object> foreignKeysParent = analyzer.getFkValues(m);
					foreignKeysParent.put(Drafts.IS_ACTIVE_ENTITY, false);

					// therefore, the select here only selects the parent's "DraftAdministrativeData_DraftUUID" column
					// from the parent with its ID as specified as the FK in the child
					tmpResult = read(Select.from(parentEntity).matching(foreignKeysParent).columns(DRAFT_ADMINISTRATIVE_DATA_DRAFT_UUID), context);
					
					// if 'lookupResultList' contains more than one entry due to several unmanaged compositions defined in the parent's model
					// the parent entity in all entries refers to the same, therefore the query would produce a result for each 'lookupResultList'
					// entry and thus 'numParents' would be more than 1 if we had not checked for unique parents before entering the if clause
					numParents += tmpResult.rowCount();
					if (tmpResult.first().isPresent()) {
						result = tmpResult;
					}
					if (numParents > 1) {
						break;
					}
				}
			}
		}

		if (numParents > 1) {
			throw new ErrorStatusException(CdsErrorStatuses.MULTIPLE_PARENTS, numParents, context.getTarget().getQualifiedName());
		}
		if (numParents == 0 || result == null) {
			throw new ErrorStatusException(CdsErrorStatuses.PARENT_NOT_EXISTING, context.getTarget().getQualifiedName());
		}
		return result.single().as(Drafts.class).getDraftAdministrativeDataDraftUuid();
	}

	/**
	 * Adds the draft fields to the result and removes associations
	 * and compositions to have a result similar to the Node.js stack.
	 * @param result the result to adapt
	 */
	private static void adaptSaveResult(Result result, CdsEntity entity) {
		result.forEach(r -> {
			removeCompositionsAndAssociations(r, entity);
		});
	}

	private static void removeCompositionsAndAssociations(Row r, CdsEntity entity) {
		Iterator<Map.Entry<String, Object>> iter = r.entrySet().iterator();
		while(iter.hasNext()) {
			Map.Entry<String, Object> entry = iter.next();
			if ((entry.getValue() instanceof List || entry.getValue() instanceof Map)
					&& isAssociation(entry.getKey(), entity)) {
				iter.remove();
			}
		}
	}

	/**
	 * Checks if the field is an association (which includes compositions)
	 * @param field the field name
	 * @param entity the entity
	 * @return {@code true} if the field is an association
	 */
	private static boolean isAssociation(String field, CdsEntity entity) {
		return entity.findAssociation(field).isPresent();
	}

	private static Result mergeInsertResults(Result result1, Result result2) {
		if (result1 == null) {
			return result2;
		}
		if (result2 == null) {
			return result1;
		}
		List<Row> entities = new ArrayList<>();
		entities.addAll(result1.list());
		entities.addAll(result2.list());
		return ResultBuilder.insertedRows(entities).result();
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected void onGcDrafts(DraftGcEventContext context) {
		AtomicLong numCancelledDrafts = new AtomicLong(0);
		Instant threshold = Instant.now().minus(context.getCdsRuntime().getEnvironment().getCdsProperties().getDrafts().getDeletionTimeout()).truncatedTo(ChronoUnit.MILLIS);
		DraftService draftService = context.getService();
		draftService.getDefinition().entities().forEach(e -> {
			if (DraftUtils.isDraftEnabledNoChild(e) && !e.getQualifiedName().endsWith(CqnAdapter.DRAFT_SUFFIX)) {
				CqnDelete deleteOldDrafts = Delete.from(e).where(c -> c.get(Drafts.IS_ACTIVE_ENTITY).eq(false).and(c.exists(
						outer -> Select.from(DraftAdministrativeData.CDS_NAME)
						.where(a -> a.get(DRAFT_UUID).eq(outer.get(DRAFT_ADMINISTRATIVE_DATA_DRAFT_UUID))
								.and(a.get(LAST_CHANGE_DATE_TIME).le(threshold))))));
				Result result = privileged(context).run((requestContext) -> {
					return draftService.cancelDraft(deleteOldDrafts);
				});
				if (result.rowCount() > 0) {
					log.info("Draft GC deleted {} drafts of entity '{}'", result.rowCount(), e.getQualifiedName());
					numCancelledDrafts.addAndGet(result.rowCount());
				}
			}
		});
		context.setResult(ResultBuilder.deletedRows(numCancelledDrafts.get()).result());
	}

}
