/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.messaging.service;

import java.io.IOException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.services.EventContext;
import com.sap.cds.services.ServiceCatalog;
import com.sap.cds.services.environment.CdsProperties.Messaging;
import com.sap.cds.services.impl.messaging.composite.MessagingCompositeService;
import com.sap.cds.services.impl.messaging.file.FileBasedMessagingBroker;
import com.sap.cds.services.impl.messaging.file.FileBasedMessagingService;
import com.sap.cds.services.impl.messaging.handlers.MessagingInitilizationHandler;
import com.sap.cds.services.impl.messaging.local.LocalMessagingService;
import com.sap.cds.services.messaging.MessagingService;
import com.sap.cds.services.outbox.OutboxMessageEventContext;
import com.sap.cds.services.outbox.OutboxService;
import com.sap.cds.services.runtime.CdsRuntimeConfiguration;
import com.sap.cds.services.runtime.CdsRuntimeConfigurer;
import com.sap.cds.services.utils.messaging.MessagingOutboxUtils;
import com.sap.cds.services.utils.messaging.service.AbstractMessagingService;
import com.sap.cds.services.utils.outbox.AbstractOutboxService;

public class MessagingServiceConfiguration implements CdsRuntimeConfiguration {

	private static final Logger logger = LoggerFactory.getLogger(FileBasedMessagingBroker.class);

	@Override
	public void services(CdsRuntimeConfigurer configurer) {
		Messaging config = configurer.getCdsRuntime().getEnvironment().getCdsProperties().getMessaging();

		// create the file based service instances
		config.getServicesByKind(FileBasedMessagingService.FILE_KIND).forEach(serviceConfig -> {
			if (serviceConfig.isEnabled()) {
				try {
					FileBasedMessagingService fileService = new FileBasedMessagingService(serviceConfig, configurer.getCdsRuntime());
					configurer.service(fileService);
				} catch (IOException e) {
					logger.error("File-based messaging service '{}' could not create the exchange file", serviceConfig.getName(), e);
				}
			}
		});

		// create the local service instances
		config.getServicesByKind(LocalMessagingService.LOCAL_KIND).forEach(serviceConfig -> {
			if (serviceConfig.isEnabled()) {
				LocalMessagingService localService = new LocalMessagingService(serviceConfig, configurer.getCdsRuntime());
				configurer.service(localService);
			}
		});

		// create the composite service if configuration is available
		if (!config.getRoutes().isEmpty()) {
			configurer.service(new MessagingCompositeService(MessagingService.COMPOSITE_NAME, config.getRoutes(), configurer.getCdsRuntime()));
		}
	}

	@Override
	public void eventHandlers(CdsRuntimeConfigurer configurer) {
		configurer.eventHandler(new MessagingInitilizationHandler());

		// register outbox handler for each messaging service
		ServiceCatalog catalog = configurer.getCdsRuntime().getServiceCatalog();
		catalog.getServices(AbstractMessagingService.class).forEach(messagingService -> {
			catalog.getServices(AbstractOutboxService.class).forEach(outboxService -> {
				String targetEvent = MessagingOutboxUtils.OUTBOX_MESSAGING_TARGET + messagingService.getName();
				outboxService.on(targetEvent, null, context -> publishedByOutbox(context.as(OutboxMessageEventContext.class), messagingService));
			});
		});
	}

	private void publishedByOutbox(OutboxMessageEventContext outboxContext, AbstractMessagingService messagingService) {
		EventContext context = MessagingOutboxUtils.toEventContext(outboxContext);
		context.put(OutboxService.IS_OUTBOXED, true);
		messagingService.emit(context);
	}
}
