/**************************************************************************
 * (C) 2019-2022 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.utils;

import java.util.Objects;
import java.util.function.Predicate;

import com.google.common.annotations.VisibleForTesting;
import com.sap.cds.reflect.CdsDefinition;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.model.CdsAnnotations;

/**
 * Reused delegate for handlers in a persistence or application services that must ensure
 * that elements must not be null.
 */
public class NotNullHandlerDelegate {

	@VisibleForTesting
	public static Predicate<CdsElement> requiresNotNullCheck() {
		return element -> {
			if (!ValidatorUtils.requiresNotNullCheck(element)) {
				return false;
			}
			if (!element.isNotNull() || !CdsAnnotations.ASSERT_NOTNULL.isTrue(element)) {
				// @assert.notNull: false deactivates null check
				return false;
			}
			if (CdsAnnotations.ODATA_FOREIGN_KEY_FOR.getOrValue(element, null) != null) {
				return false;
			}
			return true;
		};
	}

	public void accept(EventContext context, boolean allowMessageCollection) {
		ValidatorUtils.runNotNullCheck(context, requiresNotNullCheck(),
			ValidatorUtils.assertNotNull(context, Objects::isNull, (path, element, entity) -> {
				if (CdsModelUtils.isInternalOperationType(element.getDeclaringType().as(CdsDefinition.class))) {
					if(allowMessageCollection) {
						ValidatorUtils.handleValidationError(context, path, element, CdsErrorStatuses.MISSING_VALUE, element.getName());
					} else {
						throw new ErrorStatusException(CdsErrorStatuses.MISSING_VALUE, element.getName())
							.messageTarget(path, element);
					}
				} else {
					if (allowMessageCollection) {
						ValidatorUtils.handleValidationError(context, path, element, CdsErrorStatuses.VALUE_REQUIRED, element.getName(), entity);
					} else {
						throw new ErrorStatusException(CdsErrorStatuses.VALUE_REQUIRED, element.getName(), entity)
							.messageTarget(path, element);
					}
				}

			}));
	}
}
