/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.application;

import java.util.Arrays;
import java.util.Objects;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.services.ErrorStatuses;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.ServiceDelegator;
import com.sap.cds.services.ServiceException;
import com.sap.cds.services.application.ApplicationLifecycleService;
import com.sap.cds.services.application.ApplicationPreparedEventContext;
import com.sap.cds.services.application.ApplicationStoppedEventContext;
import com.sap.cds.services.application.ErrorResponseEventContext;
import com.sap.cds.services.application.ErrorResponseEventContext.ErrorResponse;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.messages.Message;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;

public class ApplicationLifecycleServiceImpl extends ServiceDelegator implements ApplicationLifecycleService {

	private static final Logger logger = LoggerFactory.getLogger(ApplicationLifecycleServiceImpl.class);

	public ApplicationLifecycleServiceImpl(String name) {
		super(name);
	}

	@Override
	public void applicationPrepared() {
		emit(ApplicationPreparedEventContext.create());
	}

	@Override
	public void applicationStopped() {
		emit(ApplicationStoppedEventContext.create());
	}

	@Override
	public ErrorResponse errorResponse(ServiceException exception) {
		ErrorResponseEventContext context = ErrorResponseEventContext.create();
		context.setException(Objects.requireNonNull(exception, "exception must not be null"));

		emit(context);

		// ensure defaults
		ErrorResponse result = context.getResult();
		if (result.getMessages() == null || result.getMessages().isEmpty()) {
			logger.warn("The list of messages returned by the error response handler is empty. Falling back to default message.");
			ServiceException e = new ErrorStatusException(ErrorStatuses.SERVER_ERROR);
			Message defaultMessage = Message.create(e.getLocalizedMessage(), e);
			result.setMessages(Arrays.asList(defaultMessage));
			result.setHttpStatus(500);
		}

		if (result.getHttpStatus() == null || result.getHttpStatus() < 400) {
			result.setHttpStatus(500);
		}
		return result;
	}

	@On(event = {EVENT_APPLICATION_PREPARED, EVENT_APPLICATION_STOPPED})
	@HandlerOrder(OrderConstants.On.AUTO_COMPLETE)
	private void autoComplete(EventContext context) {
		context.setCompleted();
	}
}
