/*********************************************************************
 * (C) 2021 SAP SE or an SAP affiliate company. All rights reserved. *
 *********************************************************************/
package com.sap.cds.services.impl.auditlog;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import com.sap.cds.CdsData;
import com.sap.cds.Result;
import com.sap.cds.Row;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.StructuredType;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnSelectListItem;
import com.sap.cds.ql.cqn.CqnUpdate;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.services.auditlog.AuditLogService;
import com.sap.cds.services.auditlog.DataModification;
import com.sap.cds.services.cds.CdsCreateEventContext;
import com.sap.cds.services.cds.CdsDeleteEventContext;
import com.sap.cds.services.cds.CdsUpdateEventContext;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.After;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.draft.CqnAdapter;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.model.CdsAnnotations;
import com.sap.cds.util.CqnStatementUtils;

/**
 * This handler is registered on all CUD events of the {@link PersistenceService}. It detects accesses and modifications of personal data
 * and logs them to the {@link AuditLogService}. Entity fields can be annotated with
 * {@link CdsAnnotations#PERSONALDATA_POTENTIALLYPERSONAL}, {@link CdsAnnotations#PERSONALDATA_POTENTIALLYSENSITIVE} to mark them as
 * personal data.
 *
 * @see <a href="https://pages.github.tools.sap/cap/docs/guides/data-privacy#indicate-privacy">Indicate Personal Data In Your Domain
 *      Model</a>
 * @see <a href="https://pages.github.tools.sap/cap/docs/guides/data-privacy#audit-log">Audit Logging</a>
 */
@ServiceName(value = "*", type = PersistenceService.class)
class PersistenceServicePersonalDataHandler implements EventHandler {
	private static final String PD_ANALYZER_MODIFICATION_KEY = PersonalDataAnalyzerModification.class.getCanonicalName();

	private final PersonalDataCaches caches;
	private final CdsRuntime runtime;
	private final AuditLogService auditLog;

	PersistenceServicePersonalDataHandler(PersonalDataCaches caches, CdsRuntime runtime) {
		this.caches = Objects.requireNonNull(caches, "caches must not be null");
		this.runtime = Objects.requireNonNull(runtime, "runtime must not be null");
		this.auditLog = runtime.getServiceCatalog().getService(AuditLogService.class, AuditLogService.DEFAULT_NAME);
	}

	// create entity on persistence service

	@After
	@HandlerOrder(OrderConstants.After.AUDIT)
	protected void afterCreate(CdsCreateEventContext context) {
		if (context.getResult() != null) {
			CdsEntity entity = context.getTarget();

			// initialise analyzer with entity and data
			PersonalDataAnalyzerModification pdAnalyzer = new PersonalDataAnalyzerModification(entity, context.getResult().list(), this.caches, this.runtime);

			// check if created entity instance contains personal data
			if (pdAnalyzer.hasPersonalData()) {
				List<CdsData> data = new ArrayList<>();

				for (CdsData row : context.getResult().list()) {
					// TODO: Handle general cqn statements
					// TODO: improve performance
					Map<String, Object> matcher = this.caches.getMeta(entity).getKeys(row);
					Select<StructuredType<?>> select = Select.from(entity).matching(matcher)
							.columns(getItemsForUpdatedData(entity, Arrays.asList(row)));
					Result result = context.getService().run(select);
					data.addAll(result.list());
				}

				// store new data in analyzer, required to create data modification objects
				pdAnalyzer.setData(data);

				List<DataModification> modifications = pdAnalyzer.getDataModifications();
				if (!modifications.isEmpty()) {
					this.auditLog.logDataModification(modifications);
				}
			}
		}
	}

	// update entity on persistence service

	@Before
	@HandlerOrder(OrderConstants.Before.AUDIT)
	protected void beforeUpdate(CdsUpdateEventContext context, List<CdsData> newData) {
		// nothing to audit if there are no modified data
		if (!newData.isEmpty()) {
			CdsEntity entity = context.getTarget();

			// initialise analyzer with entity and new data
			PersonalDataAnalyzerModification pdAnalyzer = new PersonalDataAnalyzerModification(entity, newData, this.caches, this.runtime);

			// check if updated data of entity instance contains personal data
			// TODO: evaluate rows containing personal data
			if (pdAnalyzer.hasPersonalData()) {
				List<CdsData> oldData = new ArrayList<>();

				for (CdsData row : newData) {
					// TODO: improve performance, create a batch statement
					Select<?> select = createSelectFromUpdate(context, row);
					Result result = context.getService().run(select);
					oldData.addAll(result.list());
				}

				// store old data in analyzer, required to create data modification objects later
				pdAnalyzer.setOldData(oldData);

				context.put(PD_ANALYZER_MODIFICATION_KEY, pdAnalyzer);
			}
		}
	}

	@After
	@HandlerOrder(OrderConstants.After.AUDIT)
	protected void afterUpdate(CdsUpdateEventContext context) {
		PersonalDataAnalyzerModification pdAnalyzer = (PersonalDataAnalyzerModification) context.get(PD_ANALYZER_MODIFICATION_KEY);

		// if rowCount == 0, no data is updated
		if (pdAnalyzer != null && context.getResult().rowCount() > 0) {
			context.put(PD_ANALYZER_MODIFICATION_KEY, null);

			List<CdsData> data = new ArrayList<>();

			for (CdsData row : context.getResult().list()) {
				// TODO: improve performance, create a batch statement
				// TODO: if key / values are already available in result, avoid reading updated data again
				Select<?> select = createSelectFromUpdate(context, row);
				Result result = context.getService().run(select);
				data.addAll(result.list());
			}

			// store updated data in analyzer, required to create data modification objects
			pdAnalyzer.setData(data);

			List<DataModification> modifications = pdAnalyzer.getDataModifications();
			if (!modifications.isEmpty()) {
				this.auditLog.logDataModification(modifications);
			}
		}
	}

	// delete entity on persistence service

	@Before
	@HandlerOrder(OrderConstants.Before.AUDIT)
	protected void beforeDelete(CdsDeleteEventContext context) {
		CdsEntity entity = context.getTarget();

		// first check if the target entity, a composition or an association with cascade delete has personal data
		if (this.caches.getUtils(entity).hasPersonalDataDeleted()) {
			PersonalDataAnalyzerModification pdAnalyzer = new PersonalDataAnalyzerModification(entity, this.caches, this.runtime);

			// convert delete statement into select statement using same ref and where
			CqnSelect selectEntity = CqnAdapter.toSelect(context.getCqn()).columns(this.caches.getUtils(entity).expandCompositionsForDelete());

			// enhance select statement to read also id of data subject
			PersonalDataModifier modifier = new PersonalDataModifier(entity, true, this.caches);
			CqnSelect select = CQL.copy(modifySelectStatement(entity, selectEntity), modifier);

			List<Row> oldData = context.getService().run(select).list();

			// store old data in analyzer, required to create data modification objects
			pdAnalyzer.setOldData(oldData);

			context.put(PD_ANALYZER_MODIFICATION_KEY, pdAnalyzer);
		}
	}

	@After
	@HandlerOrder(OrderConstants.After.AUDIT)
	protected void afterDelete(CdsDeleteEventContext context) {
		PersonalDataAnalyzerModification pdAnalyzer = (PersonalDataAnalyzerModification) context.get(PD_ANALYZER_MODIFICATION_KEY);

		if (pdAnalyzer != null) {
			context.put(PD_ANALYZER_MODIFICATION_KEY, null);

			List<DataModification> modifications = pdAnalyzer.getDataModifications();

			if (!modifications.isEmpty()) {
				this.auditLog.logDataModification(modifications);
			}
		}
	}

	// private helpers

	private Select<?> createSelectFromUpdate(CdsUpdateEventContext context, CdsData row) {
		CdsEntity entity = context.getTarget();

		Map<String, Object> matcher = this.caches.getMeta(entity).getKeys(row);
		Select<?> select;
		List<CqnSelectListItem> itemsToSelect = getItemsForUpdatedData(entity, Arrays.asList(row));
		if (!matcher.isEmpty()) {
			// keys are contained in data -> use matching
			select = Select.from(entity).matching(matcher).columns(itemsToSelect);
		} else {
			CqnUpdate update = context.getCqn();
			select = Select.from(update.ref()).columns(itemsToSelect);
			update.where().ifPresent(select::where);
		}
		return select;
	}

	/**
	 * @param entity   the updated entity
	 * @param dataList the data
	 * @return a list of {@link CqnSelectListItem} that contains all items that are being updated as well as the data subject ids and keys
	 */
	@SuppressWarnings("unchecked")
	private List<CqnSelectListItem> getItemsForUpdatedData(CdsEntity entity, Iterable<? extends Map<String, Object>> dataList) {
		List<CqnSelectListItem> items = new LinkedList<>();
		Set<String> fieldsToSelect = new HashSet<>();
		// add all keys
		entity.keyElements().forEach(key -> fieldsToSelect.add(key.getName()));
		// add data subject ids
		this.caches.getUtils(entity).resolveDataSubjectId().forEach(items::add);
		Map<String, List<Map<String, Object>>> associations = new HashMap<>();

		for (Map<String, Object> dataRow : dataList) {
			for (Map.Entry<String, Object> entry : dataRow.entrySet()) {
				CdsType type = entity.getElement(entry.getKey()).getType();
				if (type.isSimple() || type.isArrayed()) {
					Set<String> personalDataFields = this.caches.getMeta(entity).getPersonalDataNames();
					// add updated field
					if (personalDataFields.contains(entry.getKey())) {
						fieldsToSelect.add(entry.getKey());
					}
				} else if (type.isAssociation()) {
					// store all nested data to traverse into it recursively
					if (type.as(CdsAssociationType.class).getCardinality().getTargetMax().equals("*")) {
						if (!associations.containsKey(entry.getKey())) {
							associations.put(entry.getKey(), new ArrayList<>((List<Map<String, Object>>) entry.getValue()));
						} else {
							associations.get(entry.getKey()).addAll((List<Map<String, Object>>) entry.getValue());
						}
					} else {
						if (!associations.containsKey(entry.getKey())) {
							associations.put(entry.getKey(), Arrays.asList((Map<String, Object>) entry.getValue()));
						} else {
							associations.get(entry.getKey()).add((Map<String, Object>) entry.getValue());
						}
					}
				}
			}
		}
		fieldsToSelect.forEach(field -> items.add(CQL.get(field)));

		// traverse into the nested data
		associations.forEach((assoc, entries) -> {
			List<CqnSelectListItem> toExpand = getItemsForUpdatedData(entity.getTargetOf(assoc), entries);
			if (toExpand != null) {
				items.add(CQL.to(assoc).expand(toExpand));
			}
		});

		return items;
	}

	// static helpers

	private static CqnSelect modifySelectStatement(CdsEntity entity, CqnSelect select) {
		CqnSelect newSelect = CqnStatementUtils.resolveKeyPlaceholder(entity, select);
		newSelect = CqnStatementUtils.resolveStar(newSelect, entity);
		return newSelect;
	}

}
