/*********************************************************************
 * (C) 2021 SAP SE or an SAP affiliate company. All rights reserved. *
 *********************************************************************/
package com.sap.cds.services.impl.auditlog;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.StreamSupport;

import org.apache.commons.lang3.StringUtils;

import com.sap.cds.CdsData;
import com.sap.cds.impl.DataProcessor;
import com.sap.cds.ql.cqn.Path;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.services.auditlog.Action;
import com.sap.cds.services.auditlog.ChangedAttribute;
import com.sap.cds.services.auditlog.DataModification;
import com.sap.cds.services.runtime.CdsRuntime;

class PersonalDataAnalyzerModification extends PersonalDataAnalyzer {
	private List<? extends Map<String, Object>> oldData;

	PersonalDataAnalyzerModification(CdsStructuredType entity, PersonalDataCaches caches, CdsRuntime runtime) {
		super(entity, caches, runtime);
	}

	PersonalDataAnalyzerModification(CdsEntity entity, List<? extends Map<String, Object>> data, PersonalDataCaches caches,
			CdsRuntime runtime) {
		this(entity, caches, runtime);
		setData(data);
	}

	/**
	 * Sets the {@link List} with old or deleted data for current entity.
	 *
	 * @param oldData a {@link List} with old or deleted data
	 */
	public void setOldData(List<? extends CdsData> oldData) {
		this.oldData = oldData;
	}

	/**
	 * @return a {@link List} with modifications to personal data fields.
	 */
	public List<DataModification> getDataModifications() {
		if (this.oldData != null) {
			if (this.data == null) {
				return getDeleteDataModifications(super.entity, this.oldData);
			} else {
				return getUpdateDataModifications(super.entity, this.oldData, super.data);
			}
		} else {
			if (this.data != null) {
				return getCreateDataModification(super.entity, super.data);
			}
		}
		return Collections.emptyList();
	}

	/**
	 * Indicates whether the given {@link CdsEntity entity} contains personal data or not.
	 *
	 * @return <code>true</code> if given {@link CdsEntity entity} contains personal data.
	 */
	@Override
	public boolean hasPersonalData() {
		if (super.data != null) {
			return hasPersonalData(super.entity, super.data);
		}

		return getMeta(super.entity).hasPersonalData();
	}

	/**
	 * Indicates whether the given data contains personal data or not.
	 *
	 * @param entity the {@link CdsStructuredType entity} whose personal data fields will be determined
	 * @param data   the data that be checked for personal data
	 * @return <code>true</code> if personal data is found
	 */
	private boolean hasPersonalData(CdsStructuredType entity, Iterable<? extends Map<String, Object>> data) {
		AtomicBoolean hasPersonalData = new AtomicBoolean(false);

		DataProcessor.create().action(new DataProcessor.Action() {
			@Override
			public void entries(Path path, CdsElement element, CdsStructuredType type, Iterable<Map<String, Object>> dataList) {
				PersonalDataMeta meta = getMeta(type);

				if (meta != null) {
					Set<String> personal = meta.getPersonalDataNames();

					// check if we found already personal data in the provided data tree
					if (!hasPersonalData.get() && meta.hasPersonalData() && StreamSupport.stream(dataList.spliterator(), false)
							.anyMatch(dataRow -> dataRow.keySet().stream().anyMatch(personal::contains))) {
						hasPersonalData.set(true);
					}
				}
			}
		}).process(data, entity);

		return hasPersonalData.get();
	}

	/**
	 * Returns a list with {@link DataModification} if current entity was created.
	 *
	 * @return a list with {@link DataModification}, can be empty but not null
	 */
	private List<DataModification> getCreateDataModification(CdsStructuredType entity, List<? extends Map<String, Object>> data) {
		List<DataModification> modifications = new ArrayList<>();

		DataProcessor.create().action(new DataProcessor.Action() {
			@Override
			public void entries(Path path, CdsElement element, CdsStructuredType currentEntity, Iterable<Map<String, Object>> dataList) {
				PersonalDataMeta meta = getMeta(currentEntity);

				if (meta != null && meta.isAuditInsert()) {
					for (Map<String, Object> dataRow : dataList) {
						List<ChangedAttribute> attributes = createChangedAttributes(null, dataRow, meta);

						// ensure there are changed attributes, otherwise nothing to audit
						if (!attributes.isEmpty()) {
							DataModification modification = DataModification.create();
							modification.setAction(Action.CREATE);
							modification.setDataObject(createDataObject(dataRow, meta));
							modification.setAttributes(attributes);
							modification.setDataSubject(createDataSubject(dataRow, meta));
							modifications.add(modification);
						}
					}
				}
			}
		}).process(data, entity);

		return modifications;
	}

	/**
	 * Returns a list with {@link DataModification} if current entity was updated.
	 *
	 * @return a list with {@link DataModification}, can be empty but not null
	 */
	@SuppressWarnings("unchecked")
	private List<DataModification> getUpdateDataModifications(CdsStructuredType currentEntity,
			Iterable<? extends Map<String, Object>> oldData, Iterable<? extends Map<String, Object>> newData) {
		List<DataModification> modifications = new ArrayList<>();

		PersonalDataMeta meta = getMeta(currentEntity);

		Map<String, Map<String, Object>> keyMappedNewData = new HashMap<>();

		newData.forEach(newDataRow -> {
			String keyValues = createKeyValues(meta.getKeyNames(), newDataRow);
			keyMappedNewData.put(keyValues, newDataRow);
		});

		oldData.forEach(oldDataRow -> {
			String keyValues = createKeyValues(meta.getKeyNames(), oldDataRow);
			final Map<String, Object> newDataRow = keyMappedNewData.remove(keyValues);

			if (newDataRow != null) {
				if (meta.isAuditUpdate()) {
					List<ChangedAttribute> attributes = createChangedAttributes(oldDataRow, newDataRow, meta);

					// ensure there are changed attributes, otherwise nothing to audit
					if (!attributes.isEmpty()) {
						DataModification modification = DataModification.create();
						modification.setAction(Action.UPDATE);
						modification.setDataObject(createDataObject(oldDataRow, meta));
						modification.setDataSubject(createDataSubject(oldDataRow, meta));
						modification.setAttributes(attributes);
						modifications.add(modification);
					}
				}

				currentEntity.associations().forEach(assoc -> {
					if (oldDataRow.containsKey(assoc.getName())) {
						Object oldDataAssoc = oldDataRow.get(assoc.getName());
						Object newDataAssoc = newDataRow.get(assoc.getName());
						if (oldDataAssoc instanceof Iterable) {
							modifications.addAll(getUpdateDataModifications(currentEntity.getTargetOf(assoc.getName()),
									(Iterable<? extends Map<String, Object>>) oldDataAssoc,
									(Iterable<? extends Map<String, Object>>) newDataAssoc));
						} else if (oldDataAssoc instanceof Map) {
							modifications.addAll(getUpdateDataModifications(currentEntity.getTargetOf(assoc.getName()),
									Arrays.asList((Map<String, Object>) oldDataAssoc), Arrays.asList((Map<String, Object>) newDataAssoc)));
						}
					}
				});
			} else {
				// entity was deleted
				modifications.addAll(getDeleteDataModifications(currentEntity, Arrays.asList(oldDataRow)));
			}
		});
		// newly created entities
		keyMappedNewData.values()
				.forEach(dataRow -> modifications.addAll(getCreateDataModification(currentEntity, Arrays.asList(dataRow))));

		return modifications;
	}

	/**
	 * Returns a list with {@link DataModification} if current entity was deleted.
	 *
	 * @return a list with {@link DataModification}, can be empty but not null
	 */
	private List<DataModification> getDeleteDataModifications(CdsStructuredType entity, List<? extends Map<String, Object>> data) {
		List<DataModification> modifications = new ArrayList<>();

		DataProcessor.create().action(new DataProcessor.Action() {
			@Override
			public void entries(Path path, CdsElement element, CdsStructuredType currentEntity, Iterable<Map<String, Object>> dataList) {
				PersonalDataMeta meta = getMeta(currentEntity);

				if (meta != null && meta.isAuditDelete()) {
					for (Map<String, Object> dataRow : dataList) {
						List<ChangedAttribute> attributes = createDeletedAttributes(dataRow, meta);

						// ensure there are changed attributes, otherwise nothing to audit
						if (!attributes.isEmpty()) {
							DataModification modification = DataModification.create();
							modification.setAction(Action.DELETE);
							modification.setDataObject(createDataObject(dataRow, meta));
							modification.setDataSubject(createDataSubject(dataRow, meta));
							modification.setAttributes(attributes);
							modifications.add(modification);
						}
					}
				}
			}
		}).process(data, entity);

		return modifications;
	}

	// static helpers

	private static List<ChangedAttribute> createChangedAttributes(Map<String, Object> oldData, Map<String, Object> newData,
			PersonalDataMeta meta) {
		List<ChangedAttribute> attributes = new ArrayList<>();

		meta.getPersonalDataNames().forEach(pdName -> {
			ChangedAttribute attribute = ChangedAttribute.create();
			attribute.setName(pdName);
			Object newValue = newData.getOrDefault(pdName, null);
			if (newValue != null) {
				attribute.setNewValue(newValue.toString());
				if (oldData != null) {
					Object oldValue = oldData.getOrDefault(pdName, null);
					if (oldValue != null) {
						attribute.setOldValue(oldData.get(pdName).toString());
					}
				}
			}
			if (!StringUtils.equals(attribute.getNewValue(), attribute.getOldValue())) {
				attributes.add(attribute);
			}
		});

		return attributes;
	}

	private static List<ChangedAttribute> createDeletedAttributes(Map<String, Object> dataRow, PersonalDataMeta meta) {
		List<ChangedAttribute> attributes = new ArrayList<>();

		meta.getPersonalDataNames().forEach(pdName -> {
			ChangedAttribute attribute = ChangedAttribute.create();
			attribute.setName(pdName);
			attribute.setNewValue(null);
			if (dataRow != null) {
				Object oldValue = dataRow.getOrDefault(pdName, null);
				if (oldValue != null) {
					attribute.setOldValue(dataRow.get(pdName).toString());
				}
			}
			if (!StringUtils.equals(attribute.getNewValue(), attribute.getOldValue())) {
				attributes.add(attribute);
			}
		});

		return attributes;
	}

	private static String createKeyValues(List<String> keyNames, Map<String, Object> dataRow) {
		StringBuilder keyBuilder = new StringBuilder();
		keyNames.stream().forEach(keyName -> keyBuilder.append(dataRow.get(keyName)));
		return keyBuilder.toString();
	}
}
