/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.cds;

import java.util.List;
import java.util.Map;

import com.sap.cds.Result;
import com.sap.cds.impl.builder.model.ExpressionImpl;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Expand;
import com.sap.cds.ql.RefSegment;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.StructuredType;
import com.sap.cds.ql.StructuredTypeRef;
import com.sap.cds.ql.cqn.CqnAnalyzer;
import com.sap.cds.ql.cqn.CqnSelectListItem;
import com.sap.cds.ql.cqn.CqnStatement;
import com.sap.cds.ql.cqn.ResolvedSegment;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.utils.PathAwareCqnModifier;
import com.sap.cds.services.impl.utils.TargetAwareCqnModifier;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.model.CdsAnnotations;

@ServiceName(value = "*", type = ApplicationService.class)
public class SingletonHandler implements EventHandler {

	@Before(event = { CqnService.EVENT_READ, CqnService.EVENT_CREATE, CqnService.EVENT_UPDATE, CqnService.EVENT_UPSERT, CqnService.EVENT_DELETE })
	@HandlerOrder(OrderConstants.Before.FILTER_FIELDS)
	public void handleSingletons(EventContext context) {
		CqnStatement cqn = (CqnStatement) context.get("cqn");
		ModificationMode mode = ModificationMode.ALL;
		if(cqn.isSelect() || cqn.isInsert()) {
			mode = ModificationMode.IGNORE_LAST;
		} else if (cqn.isUpsert()) {
			mode = ModificationMode.MOVE_LAST;
		}

		SingletonModifier modifier = new SingletonModifier(CqnAnalyzer.create(context.getModel()), (CqnService) context.getService(), context.getTarget(), mode);
		CqnStatement copy = CQL.copy(cqn, modifier);

		// set limit if no limit is in the CQN so far
		if(copy.isSelect() && isSingleton(context.getTarget())) {
			((Select<?>) copy).limit(1);
		}

		// move keys to data in case of upsert, as they are not allowed as segment filter
		if(copy.isUpsert() && modifier.getKeysToMove() != null) {
			List<Map<String, Object>> entries = copy.asUpsert().entries();
			if(entries.size() != 1) {
				throw new ErrorStatusException(CdsErrorStatuses.MULTIPLE_SINGLETON_ENTRIES);
			} else {
				entries.get(0).putAll(modifier.getKeysToMove());
			}
		}

		context.put("cqn", copy);
	}

	private static boolean isSingleton(CdsEntity entity) {
		return CdsAnnotations.SINGLETON.isTrue(entity);
	}

	private enum ModificationMode {
		ALL,
		IGNORE_LAST,
		MOVE_LAST;
	}

	private static class SingletonModifier extends PathAwareCqnModifier {

		private final CqnService service;
		private final ModificationMode mode;
		private Map<String, Object> keysToMove;

		public SingletonModifier(CqnAnalyzer analyzer, CqnService service, CdsEntity target, ModificationMode mode) {
			super(analyzer, target);
			this.service = service;
			this.mode = mode;
		}

		@Override
		protected void segment(RefSegment segment, ResolvedSegment resolved, int position, StructuredTypeRef ref) {
			boolean isLast = position == (ref.size() - 1);
			if(isLast && mode == ModificationMode.IGNORE_LAST) {
				// no need to resolve last segment in case of select or insert queries
				return;
			}

			// singletons in path expressions need to be resolved by selecting their key explicitly
			if(isSingleton(resolved.entity())) {
				String[] keyElements = resolved.entity().keyElements().map(e -> e.getName()).toArray((size) -> new String[size]);
				if(keyElements.length == 0) {
					// no need to resolve singletons without keys
					return;
				}

				// check if all keys are already set
				if(resolved.keyValues().size() == resolved.entity().keyElements().count()) {
					// no need to resolve if keys are already in the segment filter
					return;
				}

				// construct partial path expression to current singleton
				StructuredType<?> singletonRef = CQL.to(ref.segments().subList(0, position + 1));
				// add key conditions to segments filter condition
				Result result = service.run(Select.from(singletonRef).columns(keyElements));
				if(isLast && mode == ModificationMode.MOVE_LAST) {
					// if no keys are found, this will result in an insert
					if(result.first().isPresent()) {
						this.keysToMove = result.single();
					}
				} else {
					if(result.rowCount() == 1) {
						Map<String, Object> singletonKeys = result.single();
						segment.filter(ExpressionImpl.matching(singletonKeys));
					} else {
						segment.filter(CQL.constant(true).eq(CQL.constant(false)));
					}
				}
			}
		}

		@Override
		public CqnSelectListItem expand(Expand<?> expand) {
			CdsEntity target = getRefTarget(expand.ref());
			if(target != null && isSingleton(target)) {
				// limit to one if entity is a singleton
				expand.limit(1);
			}
			// return original limit, if entity is not a singleton
			return expand;
		}

		@Override
		protected TargetAwareCqnModifier create(CdsEntity target) {
			return new SingletonModifier(analyzer, service, target, mode);
		}

		public Map<String, Object> getKeysToMove() {
			return keysToMove;
		}

	}

}
