/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.draft;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import com.sap.cds.ql.CQL;
import com.sap.cds.ql.ElementRef;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnSelectList;
import com.sap.cds.ql.cqn.CqnSelectListItem;
import com.sap.cds.ql.cqn.CqnSelectListValue;
import com.sap.cds.ql.cqn.CqnSortSpecification;
import com.sap.cds.ql.cqn.CqnStar;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.ql.cqn.Modifier;
import com.sap.cds.ql.impl.ExpressionVisitor;
import com.sap.cds.ql.impl.SelectListValueBuilder;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.draft.DraftAdministrativeData;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.DraftUtils;
import com.sap.cds.services.utils.ErrorStatusException;

/**
 * A {@link Modifier} to adapt references to draft fields in columns
 */
public class ColumnsModifier implements Modifier {

	private final boolean forDraftEntity;
	private final CdsEntity target;
	private final EventContext context;

	public ColumnsModifier(CdsEntity target, EventContext context) {
		this.target = target;
		this.forDraftEntity = target.getQualifiedName().endsWith(CqnAdapter.DRAFT_SUFFIX);
		this.context = context;
	}

	@Override
	public List<CqnSelectListItem> items(List<CqnSelectListItem> items) {
		if (items.isEmpty()) {
			if ((!forDraftEntity && DraftUtils.isDraftEnabled(target)) || DraftAdministrativeData.ENTITY_NAME.equals(target.getName())) {
				items.add(CqnStar.star());
			}
		}
		if (items.stream().anyMatch(i -> i.isStar())) {
			if (!forDraftEntity && DraftUtils.isDraftEnabled(target)) {
				items.add(element(Drafts.IS_ACTIVE_ENTITY));
				items.add(element(Drafts.HAS_DRAFT_ENTITY));
				items.add(element(Drafts.HAS_ACTIVE_ENTITY));
				items.add(element(Drafts.DRAFT_ADMINISTRATIVE_DATA_DRAFT_UUID));
			}
			if (DraftAdministrativeData.ENTITY_NAME.equals(target.getName())) {
				items.add(element(DraftAdministrativeData.IN_PROCESS_BY_USER));
				items.add(element(DraftAdministrativeData.DRAFT_IS_CREATED_BY_ME));
			}
		}

		items = adaptItems(items);

		return Modifier.super.items(items);
	}

	private CqnSelectListValue element(String field) {
		return CQL.get(field).withoutAlias();
	}

	@Override
	public List<CqnValue> groupBy(List<CqnValue> groupBy) {
		List<CqnValue> adapted = groupBy.stream().map(this::adaptValue).collect(Collectors.toList());
		return Modifier.super.groupBy(adapted);
	}

	@Override
	public List<CqnSortSpecification> orderBy(List<CqnSortSpecification> orderBy) {
		List<CqnSortSpecification> specs = new ArrayList<>(orderBy.size());
		for (CqnSortSpecification sortSpec: orderBy) {
			CqnValue val = adaptValue(sortSpec.value());
			specs.add(CQL.sort(val, sortSpec.order()));
		}
		return Modifier.super.orderBy(specs);
	}

	private List<CqnSelectListItem> adaptItems(List<CqnSelectListItem> items) {
		int size = items.size();
		for (int i = 0; i < size; ++i) {
			CqnSelectListItem item = items.get(i);
			if (item.isValue()) {
				items.set(i, adaptSLV(item.asValue()));
			} else if (item.isSelectList()) {
				CqnSelectList selectList = item.asSelectList();
				if (DraftUtils.isDraftEnabled(target) && selectList.ref().stream().anyMatch(s -> "*".equals(s.id()))) {
					throw new ErrorStatusException(CdsErrorStatuses.UNSUPPORTED_EXPAND_ALL_IN_DRAFT);
				}
				List<List<Boolean>> followActiveAssociations = ReferenceModifier.getAssociationDirections(target, selectList.ref(), context, null);
				if (selectList.ref().size() > 1 && followActiveAssociations.size() > 1) {
					throw new ErrorStatusException(CdsErrorStatuses.UNSUPPORTED_PATH_IN_DRAFT);
				}
				items.remove(item);
				--i;
				--size;
				for (List<Boolean> followActive: followActiveAssociations) {
					CqnSelectList newSelectList = ExpressionVisitor.copy(selectList, new SelectListModifier(target, followActive, context));
					items.add(newSelectList);
				}
			}
		}
		return items;
	}

	private CqnSelectListItem adaptSLV(CqnSelectListValue slv) {
			CqnValue val = adaptValue(slv.value());

			return SelectListValueBuilder.select(val).as(slv.displayName()).build();
	}

	private CqnValue adaptValue(CqnValue val) {
		if (val.isRef()) {
			return adaptRef(val.asRef());
		}

		return val;
	}

	private CqnValue adaptRef(CqnElementRef ref) {
		// ensure that the unsecured sibling association is not used directly
		((ElementRef<?>) ref).segments().forEach(s -> {
			if (ReferenceModifier.SIBLING_UNSECURED.equals(s.id())) {
				s.id(ReferenceModifier.SIBLING);
			}
		});
		DraftModifier modifier = new DraftModifier(target, false, false, context);

		return modifier.ref((ElementRef<?>) ref);
	}

}
