/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.messaging.file;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileSystems;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.services.messaging.TopicMessageEventContext;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.messaging.service.MessagingBrokerQueueListener;
import com.sap.cds.services.utils.messaging.service.MessagingBrokerQueueListener.MessageAccess;


/**
 * Implementation of the internal file based messaging service for local environment tests.
 */
public class FileBasedMessagingBroker {

	private static final Logger logger = LoggerFactory.getLogger(FileBasedMessagingBroker.class);
	private static final Set<String> createdBrokers = new HashSet<>();

	private final File file;
	private final Set<String> subscriptions = ConcurrentHashMap.newKeySet();
	private MessagingBrokerQueueListener listener;

	private FileBasedMessagingBroker(String name, File exchangeFile) throws IOException {
		this.file = new File(exchangeFile.getCanonicalPath());

		WatchService fileWatcher = FileSystems.getDefault().newWatchService();
		Path dir = Paths.get(file.getParentFile().getAbsolutePath());
		dir.register(fileWatcher, StandardWatchEventKinds.ENTRY_MODIFY);

		new Thread(() ->  {
			WatchKey watchKey = null;
			while (true) {
				try {
					watchKey = fileWatcher.poll(10, TimeUnit.MINUTES);
					if(watchKey != null) {
						if(!watchKey.pollEvents().isEmpty()) {
							receivedRawEvent();
						}
						watchKey.reset();
					}
				} catch(InterruptedException e) {
					logger.debug("File watching timed out, restarting watcher.");
				}
			}
		}, "FileBasedBroker: " + name).start();
	}

	/**
	 * Installs the file watcher of the specified file.
	 *
	 * @param name service name
	 * @param exchangeFile file to be used for the message exchange
	 *
	 * @return the file based broker instance
	 * @throws IOException in case, the service is already connected or the file watching initialization fails
	 */
	public static FileBasedMessagingBroker connect(String name, File exchangeFile) throws IOException {
		if (createdBrokers.contains(exchangeFile.getAbsolutePath())) {
			throw new IOException(String.format("The file based messaging broker for '%s' was already created!", exchangeFile.getAbsolutePath()));
		}

		FileBasedMessagingBroker broker = new FileBasedMessagingBroker(name, exchangeFile);
		createdBrokers.add(exchangeFile.getAbsolutePath());
		return broker;
	}

	/**
	 * Emits a new message to the file.
	 *
	 * @param topic the topic the message should be send to
	 * @param messageEventContext the TopicMessageEventContext containing the message that shall be send
	 */
	public void emitMessage(String topic, TopicMessageEventContext messageEventContext) {
		try (RandomAccessFile raFile = new RandomAccessFile(file, "rw");
				FileChannel channel = raFile.getChannel();
				FileLock lock = obtainFileLock(60000, channel)) {
			MessageLine line = new MessageLine(topic, messageEventContext.getData());
			channel.write(ByteBuffer.wrap(line.toString().getBytes(StandardCharsets.UTF_8)), channel.size());
		} catch (Exception e) { // NOSONAR
			throw new ErrorStatusException(CdsErrorStatuses.EVENT_EMITTING_FAILED, topic, e);
		}
	}

	public void registerListener(MessagingBrokerQueueListener listener) {
		if(this.listener != null) {
			throw new IllegalStateException("Only one listener is expected to be registered");
		}
		this.listener = listener;
	}

	public void subscribeTopic(String topic) {
		subscriptions.add(topic);
	}

	/**
	 * Get notified when the file contents changed.
	 */
	private void receivedRawEvent() {
		List<MessageLine> toPublishLines = new ArrayList<>();
		try (RandomAccessFile raFile = new RandomAccessFile(file, "rw");
				FileChannel channel = raFile.getChannel();
				FileLock lock = obtainFileLock(60000, channel)) {
			MessageLine line;
			List<MessageLine> leftLines = new ArrayList<>();
			while ((line = readNextLine(raFile)) != null) {
				if (subscriptions.contains(line.getBrokerTopic())) {
					toPublishLines.add(line);
				} else {
					leftLines.add(line);
				}
			}

			if (!toPublishLines.isEmpty()) {
				// cleanup the file
				raFile.setLength(0);
				for (MessageLine l : leftLines) {
					raFile.write(l.toString().getBytes(StandardCharsets.UTF_8));
				}
			}
		} catch (Exception e) { // NOSONAR
			logger.error("Could not read messages from '{}'", file, e);
		} finally {
			// notify listener if any available
			if(listener != null) {
				toPublishLines.forEach(message -> {
					try {
						listener.receivedMessage(message);
					} catch (Throwable e) { // NOSONAR
						logger.error("The received message with topic '{}' could not be handled", message.getBrokerTopic(), e);
					}
				});
			}
		}
	}

	private MessageLine readNextLine(RandomAccessFile file) throws IOException {
		String line = file.readLine();
		if (line != null) {
			return new MessageLine(new String(line.getBytes(StandardCharsets.ISO_8859_1), StandardCharsets.UTF_8));
		}
		return null;
	}

	private FileLock obtainFileLock(long timeout, FileChannel channel) throws IOException, InterruptedException, TimeoutException {
		Long quitTime = System.currentTimeMillis() + timeout;
		do {
			try {
				FileLock lock = channel.tryLock();
				if (lock != null) {
					return lock;
				} else {
					Thread.sleep(1000);
				}
			} catch (OverlappingFileLockException e) { // NOSONAR
				Thread.sleep(1000);
			}
		} while (System.currentTimeMillis() < quitTime);

		throw new TimeoutException();
	}

	/**
	 * Helper for the message line representation in the file.
	 */
	private static class MessageLine implements MessageAccess {

		private final String id = UUID.randomUUID().toString();
		private final String topic;
		private final String data;

		public MessageLine(String line) throws IOException {
			String message = line.trim();
			int separator = message.indexOf(' ');
			if (separator == -1) {
				throw new IOException("Could not find separator between topic and data in message");
			}
			this.topic = message.substring(0, separator);
			this.data = message.substring(separator).trim();
		}
		private MessageLine(String topic, String data) {
			this.topic = topic;
			this.data = data;
		}

		@Override
		public String getId() {
			return id;
		}

		@Override
		public String getMessage() {
			return data;
		}

		@Override
		public String getBrokerTopic() {
			return topic;
		}

		@Override
		public void acknowledge() {
			// not used
		}

		@Override
		public String toString() {
			return topic + ' ' + data + System.lineSeparator();
		}
	}
}
