/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.persistence;

import static com.sap.cds.DataStoreConfiguration.UPSERT_STRATEGY_REPLACE;
import static java.util.Objects.isNull;

import java.util.List;
import java.util.Map;

import com.sap.cds.SessionContext;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.SessionContextUtils;
import com.sap.cds.util.DataUtils;

/**
 * Event handler for the "managed" aspect.
 */
@ServiceName(value = "*", type = PersistenceService.class)
public class ManagedAspectHandler implements EventHandler {

	/**
	 * Handler method that calculates the administrative data properties of entities
	 * with the "managed" aspect. Usually, these are the properties createdAt / createdBy
	 * and modifiedAt / modifiedBy, but it could be any property annotated with
	 * "@cds.on.insert" / "@cds.on.update" or "@odata.on.insert" / "@odata.on.update" annotations
	 * with matching values.
	 *
	 * @param context the event context
	 */
	@Before(event = { CqnService.EVENT_CREATE, CqnService.EVENT_UPDATE, CqnService.EVENT_UPSERT })
	@HandlerOrder(OrderConstants.Before.CALCULATE_FIELDS)
	@SuppressWarnings("deprecation")
	public void calculateManagedFields(EventContext context) {
		SessionContext sessionContext = SessionContextUtils.toSessionContext(context);
		if (isNull(sessionContext.getUserContext().getId())) {
			sessionContext.getUserContext().setId("anonymous");
		}
		DataUtils dataUtils = DataUtils.create(() -> sessionContext, 7);

		String event = context.getEvent();
		List<Map<String, Object>> entries = CdsServiceUtils.getEntities(context);
		CdsStructuredType struct = context.getTarget();

		if (CqnService.EVENT_CREATE.equals(event)) {
			dataUtils.processOnInsert(struct, entries);
		} else if (CqnService.EVENT_UPDATE.equals(event)) {
			dataUtils.processOnUpdate(struct, entries, true);
		} else if (CqnService.EVENT_UPSERT.equals(event)) {
			// Upsert always proceses
			dataUtils.processOnUpdate(struct, entries, true);
			if (UPSERT_STRATEGY_REPLACE.equals(
					context.getCdsRuntime().getEnvironment().getCdsProperties().getSql().getUpsert().getStrategy())) {
				dataUtils.processOnInsert(struct, entries);
			}
		}
	}
}
