/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.utils;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.services.utils.info.CdsInfo;
import com.sap.cds.services.utils.info.CdsInfo.Details;

public class BuildInfo {

	private final static Logger logger = LoggerFactory.getLogger(BuildInfo.class);

	private static class KeyInfo {
		String fileKey;
		Details details;
		KeyInfo(String fileKey, Details details) {
			this.fileKey = fileKey;
			this.details = details;
		}
	}

	abstract static class CdsPropertyInfo implements CdsInfo {

		Map<String, Object> info = new HashMap<>();

		Map<String, KeyInfo> keyMap;

		CdsPropertyInfo(String resourceLocation, Map<String, KeyInfo> keyMap) {
			this.keyMap = keyMap;
			try {
				this.info = Collections.unmodifiableMap(
						loadFrom(resourceLocation,
								key -> keyMap.entrySet().stream().filter(e -> e.getValue().fileKey.equalsIgnoreCase(key)).map(Map.Entry::getKey).findAny().orElse(null)) );

			} catch (IOException e) {
				logger.error("Failed to load info from {}", resourceLocation, e);
			}
		}

		@Override
		public Map<String, Object> info(Details details) {
			return info.entrySet().stream().filter(g -> { return keyMap.get(g.getKey()).details.ordinal() <= details.ordinal(); }
					).collect(Collectors.toMap(Entry::getKey, Entry::getValue));
		}

		private static Map<String, Object> loadFrom(String location, Function<String, String> keyMapper) throws IOException {

			Properties source = new Properties();
			InputStream is = BuildInfo.class.getResourceAsStream(location);
			if (is == null) {
				throw new IOException("Failed to load from location '" + location + "'");
			}
			source.load(is);

			Map<String, Object> result = new HashMap<>();
			for (String fullKey : source.stringPropertyNames()) {
				String key = keyMapper.apply(fullKey);
				if (key == null) {
					continue;
				}
				Object value = source.get(fullKey);
				if (value == null || value.toString().isEmpty() ) {
					continue;
				}

				Map<String, Object> map = result;
				map.put(key, value);
			}

			return result;
		}
	}

	public static class Git extends CdsPropertyInfo {

		private final static String GIT_PROPERTIES_LOCATION = "/META-INF/cds-services-git.properties";

		@SuppressWarnings("serial")
		private static Map<String, KeyInfo> gitKeyMap = new HashMap<String, KeyInfo>() {{
			put("commit.id", new KeyInfo("git.commit.id", Details.LOW));
			put("commit.time", new KeyInfo("git.commit.time", Details.HIGH));
		}};

		public Git() {
			super(GIT_PROPERTIES_LOCATION, gitKeyMap);
		}

		@Override
		public String name() {
			return "git";
		}
	}

	public static class Maven extends CdsPropertyInfo {

		private final static String BUILD_PROPERTIES_LOCATION = "/META-INF/cds-services-build-info.properties";

		@SuppressWarnings("serial")
		private static Map<String, KeyInfo> mavenKeyMap = new HashMap<String, KeyInfo>() {{
			put("version", new KeyInfo("build.version", Details.LOW));
			put("artifact", new KeyInfo("build.artifact", Details.MEDIUM));
			put("group", new KeyInfo("build.group", Details.MEDIUM));
			put("name", new KeyInfo("build.name", Details.MEDIUM));
			put("time", new KeyInfo("build.time", Details.HIGH));
			put("java.source", new KeyInfo("build.java.source", Details.HIGH));
			put("java.target", new KeyInfo("build.java.target", Details.HIGH));
		}};

		public Maven() {
			super(BUILD_PROPERTIES_LOCATION, mavenKeyMap);
		}

		@Override
		public String name() {
			return "maven";
		}

	}

	public static void log() {

		Git gitInfo = new Git();
		gitInfo.info(Details.LOW).entrySet().stream().forEach(m ->logger.info(gitInfo.name() + "." + m.getKey() + ": " + m.getValue()));

		Maven mavenInfo = new Maven();
		mavenInfo.info(Details.LOW).entrySet().stream().forEach(m ->logger.info(mavenInfo.name() + "." + m.getKey() + ": " + m.getValue()));
	}

}
