package com.sap.cds.services.impl.cds;

import java.util.Map;
import java.util.function.Function;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.Result;
import com.sap.cds.ql.cqn.CqnDelete;
import com.sap.cds.ql.cqn.CqnInsert;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnStatement;
import com.sap.cds.ql.cqn.CqnUpdate;
import com.sap.cds.ql.cqn.CqnUpsert;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsService;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.model.CdsModelUtils;
import com.sap.cds.services.utils.services.AbstractCqnService;
import com.sap.cds.util.ProjectionResolver;

public class AbstractCdsDefinedService extends AbstractCqnService {

	private final static Logger logger = LoggerFactory.getLogger(AbstractCdsDefinedService.class);
	private final String definitionName;

	protected AbstractCdsDefinedService(String name, String model, CdsRuntime runtime) {
		super(name, runtime);
		this.definitionName = model != null ? model : name;
		runtime.getCdsModel().getService(definitionName); // validate service exists
	}

	public CdsService getDefinition() {
		return RequestContext.getCurrent(runtime).getModel().getService(definitionName);
	}

	@Override
	public Result run(CqnSelect select, Map<String, Object> namedValues) {
		return select.from().isRef() ? runProjected(select, (resolved) -> super.run(resolved, namedValues)) : super.run(select, namedValues);
	}

	@Override
	public Result run(CqnInsert insert) {
		return runProjected(insert, (resolved) -> super.run(resolved));
	}

	@Override
	public Result run(CqnUpsert upsert) {
		return runProjected(upsert, (resolved) -> super.run(resolved));
	}

	@Override
	public Result run(CqnUpdate update, Iterable<Map<String, Object>> valueSets) {
		return runProjected(update, (resolved) -> super.run(resolved, valueSets));
	}

	@Override
	public Result run(CqnDelete delete, Iterable<Map<String, Object>> valueSets) {
		return runProjected(delete, (resolved) -> super.run(resolved, valueSets));
	}

	private <T extends CqnStatement> Result runProjected(T statement, Function<T, Result> run) {
		logger.debug("CQN >>{}<<", statement);

		CdsModel model = RequestContext.getCurrent(runtime).getModel();
		CdsModelUtils.getEntityPath(statement.ref(), model); // validate
		ProjectionResolver<T> resolver = ProjectionResolver.create(model, statement)
			.condition((previous, current) -> {
				// resolve until the service is reached
				CdsEntity rootEntity = CdsModelUtils.getEntityPath(current.ref(), model).rootEntity();
				return rootEntity.getQualifier().equals(definitionName);
			})
			.resolveAll();

		logger.debug("CQN (projected) >>{}<<", resolver.getResolvedStatement());

		Result result = run.apply(resolver.getResolvedStatement());
		return result == null ? result : resolver.transform(result);
	}

	@Before
	@HandlerOrder(OrderConstants.Before.CHECK_ENTITY_FITS)
	private void checkEntityFitsService(EventContext context) {
		if(context.getTarget() != null) {
			String entityQualifier = context.getTarget().getQualifier();
			String serviceName = ((AbstractCdsDefinedService) context.getService()).getDefinition().getQualifiedName();
			// DRAFT.DraftAdministrativeData must be accessible from every service
			// TODO this can be removed again once CDS4J resolves projections
			if (!entityQualifier.equals(serviceName) && !entityQualifier.equals("DRAFT")) {
				throw new ErrorStatusException(CdsErrorStatuses.MISDIRECTED_ENTITY,
						serviceName, context.getTarget().getQualifiedName());
			}
		}
	}

	@On
	@HandlerOrder(OrderConstants.On.AUTO_COMPLETE)
	private void autoCompleteDeclaredEvents(EventContext context) {
		if (getDefinition().events().anyMatch(e -> context.getEvent().equals(e.getName()))) {
			context.setCompleted();
		}
	}

}
