/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.handlerregistry.resolver;

import java.lang.reflect.Type;
import java.util.HashMap;

import com.sap.cds.services.EventContext;
import com.sap.cds.services.EventName;
import com.sap.cds.services.impl.handlerregistry.AnnotationDescriptor;
import com.sap.cds.services.impl.handlerregistry.HandlerDescriptor;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.StringUtils;

/**
 * Handles all types of {@link EventContext} based arguments in handler methods
 */
public class EventContextArgumentResolver implements ArgumentResolver {

	private final static HashMap<Type, EventContextArgumentResolver> cache = new HashMap<>();

	/**
	 * Returns a {@link EventContextArgumentResolver}, if the given type is an {@link EventContext}
	 * @param type the type
	 * @return a {@link EventContextArgumentResolver}, if the given type is an {@link EventContext}
	 */
	public static EventContextArgumentResolver createIfApplicable(Class<?> type) {
		EventContextArgumentResolver cached;
		if((cached = cache.get(type)) != null) {
			return cached;
		}

		if(EventContext.class.isAssignableFrom(type)) {
			return new EventContextArgumentResolver(type);
		}
		return null;
	}

	private final Class<? extends EventContext> eventContextType;

	@SuppressWarnings("unchecked")
	private EventContextArgumentResolver(Class<?> type) {
		this.eventContextType = (Class<? extends EventContext>) type;
		cache.put(type, this);
	}

	@Override
	public Object resolve(EventContext context) {
		return context.as(eventContextType);
	}

	/**
	 * Does basic checks if the EventContext parameter of the given MethodHandle matches the registered event(s).
	 * A specified {@link EventContext} parameter must have a {@link EventName} annotation which matches exactly the event of the Handler annotations.
	 *
	 * @param descriptor The descriptor to be checked
	 */
	@Override
	public void verifyOrThrow(HandlerDescriptor descriptor) {
		// check if the specified EventContext matches the event declaration
		EventName eventName = eventContextType.getAnnotation(EventName.class);
		if (eventName == null || StringUtils.isEmpty(eventName.value())) {
			throw new ErrorStatusException(CdsErrorStatuses.EVENT_CONTEXT_ARGUMENT_MISSING_ANNOTATION, descriptor.getMethodName(), eventContextType.getName());
		}

		if(!eventName.value().equals("*")) {
			for (AnnotationDescriptor ad : descriptor.getAnnotations()) {
				String[] events = ad.getEvents();
				// only check if events were provided -> otherwise our suggestion wins
				if(StringUtils.notEmpty(events) != null) {
					if(events.length != 1) {
						throw new ErrorStatusException(CdsErrorStatuses.EVENT_CONTEXT_ARGUMENT_MULTIPLE_EVENTS, descriptor.getMethodName(), eventContextType.getName());

					} else if( !events[0].equals(eventName.value()) ) {
						throw new ErrorStatusException(CdsErrorStatuses.EVENT_CONTEXT_ARGUMENT_MISMATCH, descriptor.getMethodName(), eventContextType.getName(), events[0]);
					}
				}
			}
		}
	}

	@Override
	public String[] indicateEvents() {
		return new String[] { eventContextType.getAnnotation(EventName.class).value() };
	}

}
