/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.messages;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Stack;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.sap.cds.impl.builder.model.StructuredTypeProxy;
import com.sap.cds.impl.parser.PathParser;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.ElementRef;
import com.sap.cds.ql.StructuredType;
import com.sap.cds.ql.cqn.CqnReference;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.CqnVisitor;
import com.sap.cds.ql.cqn.Path;
import com.sap.cds.ql.cqn.ResolvedSegment;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.services.messages.MessageTarget;

public class MessageTargetImpl implements MessageTarget {

	private final String parameter;
	private final String entity;
	private final CqnReference reference;

	private MessageTargetImpl(String parameter, String entity, CqnReference reference) {
		this.parameter = parameter;
		this.entity = entity;
		this.reference = reference;
	}

	public static MessageTarget create(Path path, CdsElement element) {
		List<Segment> segments = new ArrayList<>();

		String rootEntity = null;
		Iterator<ResolvedSegment> iterator = path.iterator();
		if (iterator.hasNext()) {
			ResolvedSegment rootSegment = iterator.next(); // skip root entity
			rootEntity = rootSegment.segment().id();
		}
		iterator.forEachRemaining(resolvedSegment -> segments.add(resolvedSegment.segment()));
		segments.addAll(PathParser.segments(element.getName()));
		CqnReference ref;
		if (element.getType().isAssociation()) {
			ref = CQL.to(segments).asRef();
		} else {
			ref = CQL.get(segments).asRef();
		}
		return new MessageTargetImpl(MessageTarget.PARAMETER_CQN, rootEntity, ref);
	}

	public static MessageTarget create(String target) {
		return new MessageTargetImpl(target, null, null);
	}

	public static MessageTarget create(String parameter, Function<StructuredType<?>, Object> path) {
		return create(parameter, "SKIP", path);
	}

	public static MessageTarget create(String parameter, String rootElement, Function<StructuredType<?>, Object> path) {
		return create(parameter, CQL.to(rootElement), path);
	}

	@SuppressWarnings("unchecked")
	public static <E extends StructuredType<E>> MessageTarget create(String parameter, Class<E> type, Function<E, Object> path) {
		return create(parameter, StructuredTypeProxy.create(type), (Function<StructuredType<?>, Object>) path);
	}

	private static MessageTarget create(String parameter, StructuredType<?> rootElement, Function<StructuredType<?>, Object> path) {
		Object p = path.apply(rootElement);
		List<Segment> segments = new ArrayList<>();

		boolean structured = false;

		if(p instanceof StructuredType) {
			StructuredType<?> type = (StructuredType<?>) p;
			segments.addAll(type.get("SKIP").segments());
			segments.remove(segments.size() - 1);
			structured = true;
		} else if(p instanceof ElementRef) {
			segments.addAll(((ElementRef<?>) p).segments());
		} else {
			// invalid specification
			return null;
		}

		Stack<String> entityName = new Stack<>();
		rootElement.accept(new CqnVisitor() {
			@Override
			public void visit(CqnStructuredTypeRef typeRef) {
				entityName.push(typeRef.firstSegment());
			}
		});

		String name = entityName.pop();
		if(name.equals(segments.get(0).id())) {
			segments.remove(0);
		}

		if (structured) {
			return new MessageTargetImpl(parameter, name, CQL.to(segments).asRef());
		} else {
			return new MessageTargetImpl(parameter, name, CQL.get(segments).asRef());
		}
	}

	@Override
	@Deprecated
	public String getPrefix() {
		return getParameter();
	}

	@Override
	@Deprecated
	public String getEntity() {
		return entity;
	}

	@Override
	@Deprecated
	public List<Segment> getPath() {
		if (reference != null) {
			return reference.segments().stream().collect(Collectors.toList());
		}
		return null; // NOSONAR
	}

	@Override
	public CqnReference getRef() {
		return reference;
	}

	@Override
	public String getParameter() {
		return parameter;
	}
}
